/*

    bttv - Bt848 frame grabber driver

    Copyright (C) 1996,97,98 Ralph  Metzler <rjkm@thp.uni-koeln.de>
			   & Marcus Metzler <mocm@thp.uni-koeln.de>
    (c) 1999-2002 Gerd Knorr <kraxel@bytesex.org>

    some v4l2 code lines are taken from Justin's bttv2 driver which is
    (c) 2000 Justin Schoeman <justin@suntiger.ee.up.ac.za>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/kdev_t.h>

#include "bttvp.h"

#include <media/v4l2-common.h>

#include <media/v4l2-ioctl.h>

#include <media/tvaudio.h>
#include <media/msp3400.h>

#include <linux/dma-mapping.h>

#include <asm/io.h>
#include <asm/byteorder.h>

#include <media/rds.h>

// ST_0203
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,24)	

#else 
// #include <media/video-buf.h>
#include "video-buf.h"
#endif

// ST_ADD_0902 for TW6801 support 
#include "techwell.h"


unsigned int bttv_num;			/* number of Bt848s in use */
struct bttv bttvs[BTTV_MAX];

unsigned int bttv_debug = 0;
unsigned int bttv_verbose = 1;
unsigned int bttv_gpio;

/* config variables */
#ifdef __BIG_ENDIAN
static unsigned int bigendian=1;
#else
static unsigned int bigendian;
#endif
static unsigned int radio[BTTV_MAX];

// ST_0927 for enable irq debug
// static unsigned int irq_debug;
static unsigned int irq_debug = 2 ;

// ST_0120
// static unsigned int gbuffers = 8;
static unsigned int gbufsize = 0x208000;
static unsigned int gbuffers = BTTV_MAX;


static int video_nr = -1;
static int radio_nr = -1;
static int vbi_nr = -1;
static int debug_latency;

static unsigned int fdsr;

/* options */
static unsigned int combfilter;
static unsigned int lumafilter;
static unsigned int automute    = 1;
static unsigned int chroma_agc;
static unsigned int adc_crush   = 1;
static unsigned int whitecrush_upper = 0xCF;
static unsigned int whitecrush_lower = 0x7F;
static unsigned int vcr_hack;
static unsigned int irq_iswitch;
static unsigned int uv_ratio    = 50;
static unsigned int full_luma_range;
static unsigned int coring;
extern int no_overlay;

/* API features (turn on/off stuff for testing) */
static unsigned int v4l2        = 1;

// ST_0126_TEST
spinlock_t 	tw_irq_lock ;
//

// ST_0213
extern int switching ;


/* insmod args */
module_param(bttv_verbose,      int, 0644);
module_param(bttv_gpio,         int, 0644);
module_param(bttv_debug,        int, 0644);
module_param(irq_debug,         int, 0644);
module_param(debug_latency,     int, 0644);

module_param(fdsr,              int, 0444);
module_param(video_nr,          int, 0444);
module_param(radio_nr,          int, 0444);
module_param(vbi_nr,            int, 0444);
module_param(gbuffers,          int, 0444);
module_param(gbufsize,          int, 0444);

module_param(v4l2,              int, 0644);
module_param(bigendian,         int, 0644);
module_param(irq_iswitch,       int, 0644);
module_param(combfilter,        int, 0444);
module_param(lumafilter,        int, 0444);
module_param(automute,          int, 0444);
module_param(chroma_agc,        int, 0444);
module_param(adc_crush,         int, 0444);
module_param(whitecrush_upper,  int, 0444);
module_param(whitecrush_lower,  int, 0444);
module_param(vcr_hack,          int, 0444);
module_param(uv_ratio,          int, 0444);
module_param(full_luma_range,   int, 0444);
module_param(coring,            int, 0444);

module_param_array(radio, int, NULL, 0444);

MODULE_PARM_DESC(radio,"The TV card supports radio, default is 0 (no)");
MODULE_PARM_DESC(bigendian,"byte order of the framebuffer, default is native endian");
MODULE_PARM_DESC(bttv_verbose,"verbose startup messages, default is 1 (yes)");
MODULE_PARM_DESC(bttv_gpio,"log gpio changes, default is 0 (no)");
MODULE_PARM_DESC(bttv_debug,"debug messages, default is 0 (no)");
MODULE_PARM_DESC(irq_debug,"irq handler debug messages, default is 0 (no)");
MODULE_PARM_DESC(gbuffers,"number of capture buffers. range 2-32, default 8");
MODULE_PARM_DESC(gbufsize,"size of the capture buffers, default is 0x208000");
MODULE_PARM_DESC(automute,"mute audio on bad/missing video signal, default is 1 (yes)");
MODULE_PARM_DESC(chroma_agc,"enables the AGC of chroma signal, default is 0 (no)");
MODULE_PARM_DESC(adc_crush,"enables the luminance ADC crush, default is 1 (yes)");
MODULE_PARM_DESC(whitecrush_upper,"sets the white crush upper value, default is 207");
MODULE_PARM_DESC(whitecrush_lower,"sets the white crush lower value, default is 127");
MODULE_PARM_DESC(vcr_hack,"enables the VCR hack (improves synch on poor VCR tapes), default is 0 (no)");
MODULE_PARM_DESC(irq_iswitch,"switch inputs in irq handler");
MODULE_PARM_DESC(uv_ratio,"ratio between u and v gains, default is 50");
MODULE_PARM_DESC(full_luma_range,"use the full luma range, default is 0 (no)");
MODULE_PARM_DESC(coring,"set the luma coring level, default is 0 (no)");

MODULE_DESCRIPTION("bttv - v4l/v4l2 driver module for bt848/878 based cards");
MODULE_AUTHOR("Ralph Metzler & Marcus Metzler & Gerd Knorr");
MODULE_LICENSE("GPL");

/* ----------------------------------------------------------------------- */
/* sysfs                                                                   */
// ST_0126_ADD
// ST_0129, Add for switching mode support 
static void
TimerProcess
(
	unsigned long		Data 
)
{

	struct bttv *btv = (struct bttv*)Data;	
	unsigned long timeout;	

	DBG( "TimerProcess ... \n" ) ;

	if( btv->m_EnableRISC == 1 ) {		

		btv->m_EnableRISC = 0 ;

		// ST_0306
		// btwrite( btv->main.dma, TW6801_VDMAP_SA ) ;

		btor( 0x3, TW6801_VDMAC ) ;	
		
		DBG_SWITCH( "Timeprocess: #%d \n", btv->c.nr ) ;		
		
	}		
	
	// ST_0309
	/*
	if( switching ) {
		timeout = jiffies + ( msecs_to_jiffies( btv->m_LoseSignalTimer) ) ;
	} else {
		timeout = jiffies + TW6801_TIMEOUT ;	
	}
	*/

	// ST_0309
	// timeout = btv->RISCTimer[btv->c.nr].expires ;
	timeout = jiffies + TW6801_TIMEOUT ;	

	mod_timer( &btv->RISCTimer, timeout ) ;	

	//	

}

//

// ST_0129, Add for switchig mode support 
//#ifdef SWITCHING_MODE

static void
tw_SwitchingChannel
(
	struct bttv *btv
)
{

	unsigned long timeout;	
	struct timeval tv;
	unsigned int Value ;
	int	NextInput = 0 ;

	DBG_SWITCH("----->tw_SwitchingChannel\n") ;

	// ST_0130
	// btand( ~(0x3), TW6801_VDMAC ) ;	
	tw_DisableDMA(btv) ;

	btv->m_LastInput = btv->m_CurrentInput ;

	// ST_0309
	// if( btv->m_SwitchFlag > 1 ) {	
	if( btv->m_SwitchFlag > 0 ) {	

		btv->m_SwitchFlag = 0 ;		
		btv->m_CurrentInput = ( btv->m_CurrentInput + 1 ) % MAX_SWITCHING_NUMBER ;
		DBG_SWITCH( "switch to next channel(%d) !!!\n", btv->m_CurrentInput ) ;	
	}		

	btwrite( 0x40|(btv->m_CurrentInput<<2), TW6801_IFORM ) ;		

	// Brightness 
	Value = btv->m_Brightness[btv->m_CurrentInput] ;
	btwrite( Value & 0xff, TW6801_BRIGHT ) ;

	// Contrast
	Value = btv->m_Contrast[btv->m_CurrentInput] ;
	btwrite( Value  & 0xff, TW6801_CONTRAST ) ;

	// Hue 
	Value = btv->m_Hue[btv->m_CurrentInput] ;
	btwrite( Value  & 0xff, TW6801_HUE ) ;

	// Saturation 
	Value = btv->m_Saturation[btv->m_CurrentInput] ;
	btwrite( Value & 0xff, TW6801_SAT_U ) ;	

	Value = btv->m_Saturation[btv->m_CurrentInput] ;
	btwrite( Value & 0xff, TW6801_SAT_V ) ;

	return ;

}


static
unsigned long
tw_irq_check_channel_status
(
	struct bttv 	*btv
)
{
	int	NextInput = 0 ;
	unsigned long LoseSignalTimer  = 0 ;
	u32	dstat ;


	DBG_SWITCH( "----->tw_irq_check_channel_status\n") ;

	dstat = btv->DecoderStatus ;
	
	if( dstat & 0x084 ) {
		
		// There is NO video source connected 
		NextInput = ( btv->m_CurrentInput + 1 ) % MAX_SWITCHING_NUMBER ;
				
		btv->m_ChannelStatus[btv->m_CurrentInput] = DISCONNECTED ;
				
		DBG_SWITCH( "(NO)dstat: %x, Device: %d / Last: %d, Current: %d, Next: %d, Retry: %d, Polling: %d \n", 
				dstat,
				btv->c.nr, 
				btv->m_LastInput,
				btv->m_CurrentInput,
				NextInput, 
				btv->m_ChannelRetryCounter[btv->m_CurrentInput],
				btv->m_ChannelPollingCounter[btv->m_CurrentInput] ) ;

		if( btv->m_ChannelRetryCounter[btv->m_CurrentInput] > 0 ) {

			DBG_SWITCH( "Device: %d-%d is disconnected, Retry: %d, Keep polling to get stable status  ...\n", 
				btv->c.nr, 
				btv->m_CurrentInput,
				btv->m_ChannelRetryCounter[btv->m_CurrentInput] ) ;	
			
			// Retry ...
			btv->m_ChannelRetryCounter[btv->m_CurrentInput] -- ;										
					
			// Wait 27 ms still 
			// ST_0309, Keep current DISCONNECT channel to get stable status 
			// LoseSignalTimer = ( LOSESIGNAL_DELAY+SWITCHING_DELAY ) ;
			LoseSignalTimer = NO_POLLING_DELAY ;	
				
		} else {

			DBG_SWITCH( "Device: %d-%d is disconnected, Retry is timeout, SWITCH to NEXT Channel: %d...\n", 
				btv->c.nr, 
				btv->m_CurrentInput,
				NextInput ) ;	
			
			// ST_0309, We got to add more time, from no signel to signel ...
			LoseSignalTimer = ( LOSESIGNAL_DELAY+SWITCHING_DELAY ) ;
					
			btv->m_ChannelPollingCounter[NextInput] = MAX_POLLING_COUNTER ;

			// Switch to next channel 
			if( btv->m_isDisableChannel[NextInput] == CHANNEL_ENABLE ) {
				// ST_0129, for VGA/D1 frame mode support 
				// btv->m_SwitchFlag = 1 ;
				btv->m_SwitchFlag ++ ;				
			}						
		
					
		}

	} else {

		NextInput = (btv->m_CurrentInput + 1 ) % MAX_SWITCHING_NUMBER ;
			
		DBG_SWITCH( "(YES)dstat: %x, Device: %d / Last: %d, Current: %d, Next: %d, Retry: %d, Polling: %d \n", 
				dstat,
				btv->c.nr, 
				btv->m_LastInput,
				btv->m_CurrentInput,
				NextInput, 
				btv->m_ChannelRetryCounter[btv->m_CurrentInput],
				btv->m_ChannelPollingCounter[btv->m_CurrentInput] ) ;
				
		btv->m_ChannelStatus[btv->m_CurrentInput] = CONNECTED ;

		// There is video source connected 
		if ( btv->m_ChannelStatus[NextInput] == DISCONNECTED ) {
					
			DBG_SWITCH( "device: %d-%d is disconnected ...\n", 
				btv->c.nr, 
				NextInput ) ;
					
			// Next input was disconnected, do NOT switch ...
			if( btv->m_ChannelPollingCounter[btv->m_CurrentInput] > 0 ) {
				
				DBG_SWITCH( "Keep polling CURRENT channel ...\n" ) ;
				// Keep "PollingCounter" times do NOT switch ... 
				btv->m_ChannelPollingCounter[btv->m_CurrentInput] -- ;
				LoseSignalTimer = NO_POLLING_DELAY ; 
					
			} else {
					
				DBG_SWITCH( "Polling timeout, Switch to NEXT channel ...\n" ) ;	
						
				// Reset the "PollingCounter" and set the flag for switching 
				btv->m_ChannelPollingCounter[NextInput] = MAX_POLLING_COUNTER ;
				btv->m_ChannelRetryCounter[NextInput] = MAX_RETRY_COUNTER ;
				btv->m_ChannelPollingCounter[btv->m_CurrentInput] = MAX_POLLING_COUNTER ;

				LoseSignalTimer = SWITCHING_DELAY ;

				if( btv->m_isDisableChannel[NextInput] == CHANNEL_ENABLE ) {
					// ST_0129, for VGA/D1 switching
					// btv->m_SwitchFlag = 1 ;
					btv->m_SwitchFlag ++  ;					
				}		
			
			}
					
		} else {
			
			// There is video source connected for next channel
			if( btv->m_isDisableChannel[NextInput] == CHANNEL_ENABLE ) {
				// ST_0129, for VGA/D1 switching
				//btv->m_SwitchFlag = 1 ;
				btv->m_SwitchFlag ++  ;									
			}						
					
			btv->m_ChannelPollingCounter[btv->m_CurrentInput] = MAX_POLLING_COUNTER ;					
			btv->m_ChannelPollingCounter[NextInput] = MAX_POLLING_COUNTER ;
			btv->m_ChannelRetryCounter[NextInput] = MAX_RETRY_COUNTER ;
			LoseSignalTimer = SWITCHING_DELAY  ;
		}
		
	}

	return( LoseSignalTimer ) ;

}

//#endif




// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
static ssize_t 
show_card
(
	struct device *cd,
	struct device_attribute *attr, 
	char *buf
)
{
	struct video_device *vfd = container_of(cd, struct video_device, dev);
	struct bttv *btv = dev_get_drvdata(vfd->parent);
	return sprintf(buf, "%d\n", btv ? btv->c.type : UNSET);
}

static DEVICE_ATTR(card, S_IRUGO, show_card, NULL);

#else
static ssize_t show_card(struct class_device *cd, char *buf)
{
	struct video_device *vfd = to_video_device(cd);
	struct bttv *btv = dev_get_drvdata(vfd->dev);
	return sprintf(buf, "%d\n", btv ? btv->c.type : UNSET);
}

static CLASS_DEVICE_ATTR(card, S_IRUGO, show_card, NULL);

#endif

/* ----------------------------------------------------------------------- */
/* static data                                                             */

/* special timing tables from conexant... */
static u8 SRAM_Table[][60] =
{
	/* PAL digital input over GPIO[7:0] */
	{
		45, // 45 bytes following
		0x36,0x11,0x01,0x00,0x90,0x02,0x05,0x10,0x04,0x16,
		0x12,0x05,0x11,0x00,0x04,0x12,0xC0,0x00,0x31,0x00,
		0x06,0x51,0x08,0x03,0x89,0x08,0x07,0xC0,0x44,0x00,
		0x81,0x01,0x01,0xA9,0x0D,0x02,0x02,0x50,0x03,0x37,
		0x37,0x00,0xAF,0x21,0x00
	},
	/* NTSC digital input over GPIO[7:0] */
	{
		51, // 51 bytes following
		0x0C,0xC0,0x00,0x00,0x90,0x02,0x03,0x10,0x03,0x06,
		0x10,0x04,0x12,0x12,0x05,0x02,0x13,0x04,0x19,0x00,
		0x04,0x39,0x00,0x06,0x59,0x08,0x03,0x83,0x08,0x07,
		0x03,0x50,0x00,0xC0,0x40,0x00,0x86,0x01,0x01,0xA6,
		0x0D,0x02,0x03,0x11,0x01,0x05,0x37,0x00,0xAC,0x21,
		0x00,
	},
	// TGB_NTSC392 // quartzsight
	// This table has been modified to be used for Fusion Rev D
	{
		0x2A, // size of table = 42
		0x06, 0x08, 0x04, 0x0a, 0xc0, 0x00, 0x18, 0x08, 0x03, 0x24,
		0x08, 0x07, 0x02, 0x90, 0x02, 0x08, 0x10, 0x04, 0x0c, 0x10,
		0x05, 0x2c, 0x11, 0x04, 0x55, 0x48, 0x00, 0x05, 0x50, 0x00,
		0xbf, 0x0c, 0x02, 0x2f, 0x3d, 0x00, 0x2f, 0x3f, 0x00, 0xc3,
		0x20, 0x00
	}
};

const struct bttv_tvnorm bttv_tvnorms[] = {
	/* PAL-BDGHI */
	/* max. active video is actually 922, but 924 is divisible by 4 and 3! */
	/* actually, max active PAL with HSCALE=0 is 948, NTSC is 768 - nil */

	// ST_ADD_0318 for VBI support
	{
		.v4l2_id        	= V4L2_STD_NTSC_M | V4L2_STD_NTSC_M_KR,
		.name           	= "NTSC",
		.Fsc            	= 28636363, 	//ST_TEST_1012 28636363,
		.swidth         	= 720,	// ST_0508	640,
		.sheight        	= 480,
		.totalwidth     	= 720,
		.adelay         	= 0x68,
		.bdelay         	= 0x5d,
		.iform          	= (TW6801_SDT_STD_NTSC_M),		
		.scaledtwidth	= 720, 	// 780,			//910,
		.hdelayx1       	= 0x10,	//20, // 0x5,  			//0x0F,			//10,			//	128,
		.hactivex1      	= 720,			// 0x2D0, 		// 910,
		.vdelay         	= 0x15, //(525-480)/2,	// 0x18,			//0x1a,
		.vbipack        	= 144, 			// 144,
		.sram           	= 1,
		.vbistart		= { 10, 21 },		
	},
	{
		.v4l2_id       	= V4L2_STD_PAL,
		.name           	= "PAL",
		.Fsc            	= 35468950, // 28636000, 	// ST_TEST_1126 35468950,
		.swidth         	= 720,
		.sheight        	= 576,
		.totalwidth     	= 720,
		.adelay         	= 0x7F, // 0x7f,
		.bdelay         	= 0x72, // 0x72,
		.iform          	= (TW6801_SDT_STD_PAL_BDGHI),
		.scaledtwidth 	= 720,
		.hdelayx1       	= 15, //0x01,
		.hactivex1      	= 720,
		.vdelay         	= 0x17,		//0x20,
		.vbipack        	= 144,
		.sram           	= 1, // 0,
		/* ITU-R frame line number of the first VBI line
		   we can capture, of the first and second field. */
		.vbistart		= { 10,21 },
	},

// ST_TEST_0318 for VBI
#if 0	
	{
		.v4l2_id        	= V4L2_STD_NTSC_M | V4L2_STD_NTSC_M_KR,
		.name           	= "NTSC",
		.Fsc            	= 28636363, //ST_TEST_1012 28636363,
		.swidth         	= 768,
		.sheight        	= 480,	//576, 	// ST_TEST_1126 480,
		.totalwidth     	= 780, 	// 910,
		.adelay         	= 0x68,
		.bdelay         	= 0x5d,

		// ST_MODIFY
		// .iform          	= (BT848_IFORM_NTSC|BT848_IFORM_XT0),
		.iform          	= (TW6801_SDT_STD_NTSC_M),		
		
		.scaledtwidth   = 780,	//910,
		.hdelayx1       	= 10,		//	128,
		.hactivex1      	= 910,
		.vdelay         	= 0x1a,
		.vbipack        	= 144, 	// 144,
		.sram           	= 1,
		// ST_1126 
		// .vbistart		= { 10, 273 },
		.vbistart		= { 10, 21 },		
	},
	
#endif

	{
		.v4l2_id        	= V4L2_STD_SECAM,
		.name           	= "SECAM",
		.Fsc            	= 35468950,
		.swidth         	= 924,
		.sheight        	= 576,
		.totalwidth     	= 1135,
		.adelay         	= 0x7f,
		.bdelay         	= 0xb0,

		// ST_MODIFY
		// .iform          	= (BT848_IFORM_SECAM|BT848_IFORM_XT1),
		.iform          	= (TW6801_SDT_STD_SECAM),		
		
		.scaledtwidth  	= 1135,
		.hdelayx1       	= 186,
		.hactivex1      	= 922,
		.vdelay         	= 0x20,
		.vbipack        	= 255,
		.sram           	= 0, /* like PAL, correct? */
		.vbistart		= { 7, 320 },
	},{
		.v4l2_id        	= V4L2_STD_PAL_Nc,
		.name           	= "PAL-Nc",
		.Fsc            	= 28636363,
		.swidth         	= 640,
		.sheight        	= 576,
		.totalwidth     	= 910,
		.adelay         	= 0x68,
		.bdelay         	= 0x5d,

		// St_MODIFY
		// .iform          	= (BT848_IFORM_PAL_NC|BT848_IFORM_XT0),
		.iform          	= (TW6801_SDT_STD_PAL_CN),		
		
		.scaledtwidth  	= 780,
		.hdelayx1       	= 130,
		.hactivex1      	= 734,
		.vdelay         	= 0x1a,
		.vbipack        	= 144,
		.sram           	= -1,
		.vbistart		= { 7, 320 },
	},{
		.v4l2_id        	= V4L2_STD_PAL_M,
		.name           	= "PAL-M",
		.Fsc            	= 28636363,
		.swidth         	= 720,
		.sheight        	= 480,
		.totalwidth     	= 910,
		.adelay         	= 0x68,
		.bdelay         	= 0x5d,

		//ST_MODIFY
		// .iform          	= (BT848_IFORM_PAL_M|BT848_IFORM_XT0),
		.iform          	= (TW6801_SDT_STD_PAL_M),		
		
		.scaledtwidth  	= 780,
		.hdelayx1       	= 135,
		.hactivex1      	= 754,
		.vdelay         	= 0x1a,
		.vbipack        	= 144,
		.sram           	= -1,
		.vbistart		= { 10, 273 },
	},{
		.v4l2_id        	= V4L2_STD_PAL_N,
		.name           	= "PAL-N",
		.Fsc            	= 35468950,
		.swidth         	= 768,
		.sheight        	= 576,
		.totalwidth     	= 1135,
		.adelay         	= 0x7f,
		.bdelay         	= 0x72,

		// ST_MODIFY
		// .iform          	= (BT848_IFORM_PAL_N|BT848_IFORM_XT1),
		.iform          	= (TW6801_SDT_STD_PAL_M),		
		
		.scaledtwidth  	= 944,
		.hdelayx1       	= 186,
		.hactivex1      	= 922,
		.vdelay         	= 0x20,
		.vbipack        	= 144,
		.sram           	= -1,
		.vbistart		= { 7, 320},
	},{
		.v4l2_id        	= V4L2_STD_NTSC_M_JP,
		.name           	= "NTSC-JP",
		.Fsc            	= 28636363,	// ST_TEST_1012 28636363,
		.swidth         	= 640,
		.sheight        	= 480,
		.totalwidth     	= 910,
		.adelay         	= 0x68,
		.bdelay         	= 0x5d,

		// ST_MODIFY
		// .iform          	= (BT848_IFORM_NTSC_J|BT848_IFORM_XT0),
		.iform          	= (TW6801_SDT_STD_NTSC_M),		
		
		.scaledtwidth  	= 780, 	// 780,
		.hdelayx1       	= 135,
		.hactivex1      	= 754,
		.vdelay         	= 0x16,
		.vbipack        	= 144,
		.sram           	= -1,
		.vbistart		= {10, 273},
	},{
		/* that one hopefully works with the strange timing
		 * which video recorders produce when playing a NTSC
		 * tape on a PAL TV ... */
		.v4l2_id        	= V4L2_STD_PAL_60,
		.name           	= "PAL-60",
		.Fsc            	= 35468950,
		.swidth         	= 924,
		.sheight        	= 480,
		.totalwidth     	= 1135,
		.adelay         	= 0x7f,
		.bdelay         	= 0x72,

		// ST_MODIFY
		// .iform          	= (BT848_IFORM_PAL_BDGHI|BT848_IFORM_XT1),
		.iform          	= (TW6801_SDT_STD_PAL_60),		
		
		.scaledtwidth  	= 1135,
		.hdelayx1       	= 186,
		.hactivex1      	= 924,
		.vdelay         	= 0x1a,
		.vbipack        	= 255,
		.vtotal         	= 524,
		.sram           	= -1,
		.vbistart		= { 10, 273 },
		
	}
		
};

static const unsigned int BTTV_TVNORMS = ARRAY_SIZE(bttv_tvnorms);

/* ----------------------------------------------------------------------- */
/* bttv format list
   packed pixel formats must come first */

#if 0
static const struct bttv_format bttv_formats[] = {
	{
		.name     = "8 bpp, gray",
		.palette  = VIDEO_PALETTE_GREY,
		.fourcc   = V4L2_PIX_FMT_GREY,
		.btformat = BT848_COLOR_FMT_Y8,
		.depth    = 8,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "8 bpp, dithered color",
		.palette  = VIDEO_PALETTE_HI240,
		.fourcc   = V4L2_PIX_FMT_HI240,
		.btformat = BT848_COLOR_FMT_RGB8,
		.depth    = 8,
		.flags    = FORMAT_FLAGS_PACKED | FORMAT_FLAGS_DITHER,
	},{
		.name     = "15 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB555,
		.fourcc   = V4L2_PIX_FMT_RGB555,
		.btformat = BT848_COLOR_FMT_RGB15,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "15 bpp RGB, be",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_RGB555X,
		.btformat = BT848_COLOR_FMT_RGB15,
		.btswap   = 0x03, /* byteswap */
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "16 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB565,
		.fourcc   = V4L2_PIX_FMT_RGB565,
		.btformat = BT848_COLOR_FMT_RGB16,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "16 bpp RGB, be",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_RGB565X,
		.btformat = BT848_COLOR_FMT_RGB16,
		.btswap   = 0x03, /* byteswap */
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "24 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB24,
		.fourcc   = V4L2_PIX_FMT_BGR24,
		.btformat = BT848_COLOR_FMT_RGB24,
		.depth    = 24,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "32 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB32,
		.fourcc   = V4L2_PIX_FMT_BGR32,
		.btformat = BT848_COLOR_FMT_RGB32,
		.depth    = 32,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "32 bpp RGB, be",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_RGB32,
		.btformat = BT848_COLOR_FMT_RGB32,
		.btswap   = 0x0f, /* byte+word swap */
		.depth    = 32,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, packed, YUYV",
		.palette  = VIDEO_PALETTE_YUV422,
		.fourcc   = V4L2_PIX_FMT_YUYV,
		.btformat = BT848_COLOR_FMT_YUY2,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, packed, YUYV",
		.palette  = VIDEO_PALETTE_YUYV,
		.fourcc   = V4L2_PIX_FMT_YUYV,
		.btformat = BT848_COLOR_FMT_YUY2,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, packed, UYVY",
		.palette  = VIDEO_PALETTE_UYVY,
		.fourcc   = V4L2_PIX_FMT_UYVY,
		.btformat = BT848_COLOR_FMT_YUY2,
		.btswap   = 0x03, /* byteswap */
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, planar, Y-Cb-Cr",
		.palette  = VIDEO_PALETTE_YUV422P,
		.fourcc   = V4L2_PIX_FMT_YUV422P,
		.btformat = BT848_COLOR_FMT_YCrCb422,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PLANAR,
		.hshift   = 1,
		.vshift   = 0,
	},{
		.name     = "4:2:0, planar, Y-Cb-Cr",
		.palette  = VIDEO_PALETTE_YUV420P,
		.fourcc   = V4L2_PIX_FMT_YUV420,
		.btformat = BT848_COLOR_FMT_YCrCb422,
		.depth    = 12,
		.flags    = FORMAT_FLAGS_PLANAR,
		.hshift   = 1,
		.vshift   = 1,
	},{
		.name     = "4:2:0, planar, Y-Cr-Cb",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_YVU420,
		.btformat = BT848_COLOR_FMT_YCrCb422,
		.depth    = 12,
		.flags    = FORMAT_FLAGS_PLANAR | FORMAT_FLAGS_CrCb,
		.hshift   = 1,
		.vshift   = 1,
	},{
		.name     = "4:1:1, planar, Y-Cb-Cr",
		.palette  = VIDEO_PALETTE_YUV411P,
		.fourcc   = V4L2_PIX_FMT_YUV411P,
		.btformat = BT848_COLOR_FMT_YCrCb411,
		.depth    = 12,
		.flags    = FORMAT_FLAGS_PLANAR,
		.hshift   = 2,
		.vshift   = 0,
	},{
		.name     = "4:1:0, planar, Y-Cb-Cr",
		.palette  = VIDEO_PALETTE_YUV410P,
		.fourcc   = V4L2_PIX_FMT_YUV410,
		.btformat = BT848_COLOR_FMT_YCrCb411,
		.depth    = 9,
		.flags    = FORMAT_FLAGS_PLANAR,
		.hshift   = 2,
		.vshift   = 2,
	},{
		.name     = "4:1:0, planar, Y-Cr-Cb",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_YVU410,
		.btformat = BT848_COLOR_FMT_YCrCb411,
		.depth    = 9,
		.flags    = FORMAT_FLAGS_PLANAR | FORMAT_FLAGS_CrCb,
		.hshift   = 2,
		.vshift   = 2,
	},{
		.name     = "raw scanlines",
		.palette  = VIDEO_PALETTE_RAW,
		.fourcc   = -1,
		.btformat = BT848_COLOR_FMT_RAW,
		.depth    = 8,
		.flags    = FORMAT_FLAGS_RAW,
	}
};

#else
// ST_ADD_1001
static const struct bttv_format bttv_formats[] = {
	{
		.name     = "15 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB555,
		.fourcc   = V4L2_PIX_FMT_RGB555,
		.btformat = TW6801_VDMAC_COLORF_RGB15,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "15 bpp RGB, be",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_RGB555X,
		.btformat = TW6801_VDMAC_COLORF_RGB15,
		.btswap   = 0x03, /* byteswap */
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "16 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB565,
		.fourcc   = V4L2_PIX_FMT_RGB565,
		.btformat = TW6801_VDMAC_COLORF_RGB16,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "16 bpp RGB, be",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_RGB565X,
		.btformat = TW6801_VDMAC_COLORF_RGB16,
		.btswap   = 0x03, /* byteswap */
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "24 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB24,
		.fourcc   = V4L2_PIX_FMT_BGR24,
		.btformat = TW6801_VDMAC_COLORF_RGB24,
		.depth    = 24,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "32 bpp RGB, le",
		.palette  = VIDEO_PALETTE_RGB32,
		.fourcc   = V4L2_PIX_FMT_BGR32,
		.btformat = TW6801_VDMAC_COLORF_RGB32,
		.depth    = 32,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "32 bpp RGB, be",
		.palette  = -1,
		.fourcc   = V4L2_PIX_FMT_RGB32,
		.btformat = TW6801_VDMAC_COLORF_RGB32,
		// .btswap   = 0x0f, /* byte+word swap */
		.depth    = 32,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, packed, YUYV",
		.palette  = VIDEO_PALETTE_YUV422,
		.fourcc   = V4L2_PIX_FMT_YUYV,
		.btformat = TW6801_VDMAC_COLORF_YUY2,
		.depth    = 16,
		.btswap   = 0x01, /* byteswap */
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, packed, YUYV",
		.palette  = VIDEO_PALETTE_YUYV,
		.fourcc   = V4L2_PIX_FMT_YUYV,
		.btformat = TW6801_VDMAC_COLORF_YUY2,
		.btswap   = 0x00, /* byteswap */
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	},{
		.name     = "4:2:2, packed, UYVY",
		.palette  = VIDEO_PALETTE_UYVY,
		.fourcc   = V4L2_PIX_FMT_UYVY,
		.btformat = TW6801_VDMAC_COLORF_YUY2,
		.btswap   = 0x01,
		.depth    = 16,
		.flags    = FORMAT_FLAGS_PACKED,
	}, 

	// ST_ADD_1026
	/*
	{
		.name     	= "4:2:0, packed, Y-Cb-Cr",
		.palette  	= VIDEO_PALETTE_YUV420P,
		.fourcc   	= V4L2_PIX_FMT_YUV420,
		.btformat 	= TW6801_VDMAC_COLORF_YUY2,
		.btswap   	= 0x00, 
		.depth    	= 12,
		.flags    	= FORMAT_FLAGS_PACKED,
	}
	*/
		
		
};


#endif

static const unsigned int BTTV_FORMATS = ARRAY_SIZE(bttv_formats);

/* ----------------------------------------------------------------------- */

#define V4L2_CID_PRIVATE_CHROMA_AGC  (V4L2_CID_PRIVATE_BASE + 0)
#define V4L2_CID_PRIVATE_COMBFILTER  (V4L2_CID_PRIVATE_BASE + 1)
#define V4L2_CID_PRIVATE_AUTOMUTE    (V4L2_CID_PRIVATE_BASE + 2)
#define V4L2_CID_PRIVATE_LUMAFILTER  (V4L2_CID_PRIVATE_BASE + 3)
#define V4L2_CID_PRIVATE_AGC_CRUSH   (V4L2_CID_PRIVATE_BASE + 4)
#define V4L2_CID_PRIVATE_VCR_HACK    (V4L2_CID_PRIVATE_BASE + 5)
#define V4L2_CID_PRIVATE_WHITECRUSH_UPPER   (V4L2_CID_PRIVATE_BASE + 6)
#define V4L2_CID_PRIVATE_WHITECRUSH_LOWER   (V4L2_CID_PRIVATE_BASE + 7)
#define V4L2_CID_PRIVATE_UV_RATIO    (V4L2_CID_PRIVATE_BASE + 8)
#define V4L2_CID_PRIVATE_FULL_LUMA_RANGE    (V4L2_CID_PRIVATE_BASE + 9)
#define V4L2_CID_PRIVATE_CORING      (V4L2_CID_PRIVATE_BASE + 10)
#define V4L2_CID_PRIVATE_LASTP1      (V4L2_CID_PRIVATE_BASE + 11)

static const struct v4l2_queryctrl no_ctl = {
	.name  = "42",
	.flags = V4L2_CTRL_FLAG_DISABLED,
};

static const struct v4l2_queryctrl bttv_ctls[] = {
	/* --- video --- */
	{
		.id            		= V4L2_CID_BRIGHTNESS,
		.name          	= "Brightness",
		.minimum       	= -127,
		.maximum       = 128,
		.step          	= 2,
		.default_value 	= 0 ,
		.type          	= V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            		= V4L2_CID_CONTRAST,
		.name          	= "Contrast",
		.minimum       	= 0,
		.maximum       = 0xFF,
		.step          	= 2,
		.default_value = 84,
		.type          	= V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            		= V4L2_CID_SATURATION,
		.name          	= "Saturation",
		.minimum       	= 0,
		.maximum       = 0xFF,
		.step          	= 2,
		.default_value 	= 0x7F,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            		= V4L2_CID_HUE,
		.name          	= "Hue",
		.minimum       	= -90,
		.maximum       = 90,
		.step          	= 1,
		.default_value 	= 0 ,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},
	/* --- audio --- */
	{
		.id            = V4L2_CID_AUDIO_MUTE,
		.name          = "Mute",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_AUDIO_VOLUME,
		.name          = "Volume",
		.minimum       = 0,
		.maximum       = 65535,
		.step          = 65535/100,
		.default_value = 65535,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            = V4L2_CID_AUDIO_BALANCE,
		.name          = "Balance",
		.minimum       = 0,
		.maximum       = 65535,
		.step          = 65535/100,
		.default_value = 32768,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            = V4L2_CID_AUDIO_BASS,
		.name          = "Bass",
		.minimum       = 0,
		.maximum       = 65535,
		.step          = 65535/100,
		.default_value = 32768,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            = V4L2_CID_AUDIO_TREBLE,
		.name          = "Treble",
		.minimum       = 0,
		.maximum       = 65535,
		.step          = 65535/100,
		.default_value = 32768,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},
	/* --- private --- */
	{
		.id            = V4L2_CID_PRIVATE_CHROMA_AGC,
		.name          = "chroma agc",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_COMBFILTER,
		.name          = "combfilter",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_AUTOMUTE,
		.name          = "automute",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_LUMAFILTER,
		.name          = "luma decimation filter",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_AGC_CRUSH,
		.name          = "agc crush",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_VCR_HACK,
		.name          = "vcr hack",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_WHITECRUSH_UPPER,
		.name          = "whitecrush upper",
		.minimum       = 0,
		.maximum       = 255,
		.step          = 1,
		.default_value = 0xCF,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            = V4L2_CID_PRIVATE_WHITECRUSH_LOWER,
		.name          = "whitecrush lower",
		.minimum       = 0,
		.maximum       = 255,
		.step          = 1,
		.default_value = 0x7F,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            = V4L2_CID_PRIVATE_UV_RATIO,
		.name          = "uv ratio",
		.minimum       = 0,
		.maximum       = 100,
		.step          = 1,
		.default_value = 50,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	},{
		.id            = V4L2_CID_PRIVATE_FULL_LUMA_RANGE,
		.name          = "full luma range",
		.minimum       = 0,
		.maximum       = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	},{
		.id            = V4L2_CID_PRIVATE_CORING,
		.name          = "coring",
		.minimum       = 0,
		.maximum       = 3,
		.step          = 1,
		.default_value = 0,
		.type          = V4L2_CTRL_TYPE_INTEGER,
	}



};
static const int BTTV_CTLS = ARRAY_SIZE(bttv_ctls);

/* ----------------------------------------------------------------------- */
/* resource management                                                     */

static
int check_alloc_btres(struct bttv *btv, struct bttv_fh *fh, int bit)
{
	if (fh->resources & bit)
		/* have it already allocated */
		return 1;

	/* is it free? */
	mutex_lock(&btv->reslock);
	if (btv->resources & bit) {
		/* no, someone else uses it */
		mutex_unlock(&btv->reslock);
		return 0;
	}
	/* it's free, grab it */
	fh->resources  |= bit;
	btv->resources |= bit;
	mutex_unlock(&btv->reslock);
	return 1;
}

static
int check_btres(struct bttv_fh *fh, int bit)
{

	DBG("check_btres :: resources: %x \n", fh->resources ) ;
	return (fh->resources & bit);
}

static
int locked_btres(struct bttv *btv, int bit)
{
	return (btv->resources & bit);
}

static
void free_btres(struct bttv *btv, struct bttv_fh *fh, int bits)
{
	if ((fh->resources & bits) != bits) {
		/* trying to free ressources not allocated by us ... */
		DBG_FPS("bttv: BUG! (btres)\n");
	}
	mutex_lock(&btv->reslock);
	fh->resources  &= ~bits;
	btv->resources &= ~bits;
	mutex_unlock(&btv->reslock);
}

/* ----------------------------------------------------------------------- */
/* If Bt848a or Bt849, use PLL for PAL/SECAM and crystal for NTSC          */

/* Frequency = (F_input / PLL_X) * PLL_I.PLL_F/PLL_C
   PLL_X = Reference pre-divider (0=1, 1=2)
   PLL_C = Post divider (0=6, 1=4)
   PLL_I = Integer input
   PLL_F = Fractional input

   F_input = 28.636363 MHz:
   PAL (CLKx2 = 35.46895 MHz): PLL_X = 1, PLL_I = 0x0E, PLL_F = 0xDCF9, PLL_C = 0
*/

static void set_pll_freq(struct bttv *btv, unsigned int fin, unsigned int fout)
{
	unsigned char fl, fh, fi;

	/* prevent overflows */
	fin/=4;
	fout/=4;

	fout*=12;
	fi=fout/fin;

	fout=(fout%fin)*256;
	fh=fout/fin;

	fout=(fout%fin)*256;
	fl=fout/fin;

	btwrite(fl, BT848_PLL_F_LO);
	btwrite(fh, BT848_PLL_F_HI);
	btwrite(fi|BT848_PLL_X, BT848_PLL_XCI);
}



static void set_pll(struct bttv *btv)
{
	int i;

	DBG("----->set_pll \n") ;
	

	if (!btv->pll.pll_crystal) {
		DBG("<-----set_pll-1\n") ;
		return;
	}

	if (btv->pll.pll_ofreq == btv->pll.pll_current) {
		dprintk("TW6801-%d: PLL: no change required\n",btv->c.nr);
		return;
	}

	if (btv->pll.pll_ifreq == btv->pll.pll_ofreq) {
		
		/* no PLL needed */
		if (btv->pll.pll_current == 0) {
			DBG("<-----set_pll-2\n") ;
			return;
		}
		
		DBG(KERN_INFO "TW6801-%d: PLL can sleep, using XTAL (%d).\n",
			btv->c.nr,btv->pll.pll_ifreq);

		// ST_MODIFY_0924 for tw6801 support 
		// btwrite(0x00,BT848_TGCTRL);
		// btwrite(0x00,BT848_PLL_XCI);
		btwrite(0x00,BT848_TGCTRL);
		btwrite(0x00,BT848_PLL_XCI);

		btv->pll.pll_current = 0;

		DBG("<-----set_pll-3\n") ;

		return;
		
	}

	DBG(KERN_INFO "TW6801-%d: PLL: %d => %d ",btv->c.nr,
		btv->pll.pll_ifreq, btv->pll.pll_ofreq);
	
	set_pll_freq(btv, btv->pll.pll_ifreq, btv->pll.pll_ofreq);

	for (i=0; i<10; i++) {
		/*  Let other people run while the PLL stabilizes */
		DBG(".");
		msleep(10);

		if (btread(BT848_DSTATUS) & BT848_DSTATUS_PLOCK) {
			btwrite(0,BT848_DSTATUS);
		} else {
			btwrite(0x08,BT848_TGCTRL);
			btv->pll.pll_current = btv->pll.pll_ofreq;
			bttv_printk(" ok\n");
			return;
		}
	}
	btv->pll.pll_current = -1;
	DBG("failed\n");
	return;
}

/* used to switch between the bt848's analog/digital video capture modes */
static void tw_set_timing(struct bttv *btv)
{
	int i, len;
	int table_idx = bttv_tvnorms[btv->tvnorm].sram;
	int fsc       = bttv_tvnorms[btv->tvnorm].Fsc;

	DBG("----->tw_set_timing\n" ) ;


	if (UNSET == bttv_tvcards[btv->c.type].muxsel[btv->input]) {
		
		DBG("TW6801-v%d: load digital timing table (table_idx=%d)\n",
			btv->c.nr,table_idx);

		/* timing change...reset timing generator address */
		btwrite(0x00, BT848_TGCTRL);
		btwrite(0x02, BT848_TGCTRL);
		btwrite(0x00, BT848_TGCTRL);

		len=SRAM_Table[table_idx][0];
		for(i = 1; i <= len; i++)
			btwrite(SRAM_Table[table_idx][i],BT848_TGLB);
		btv->pll.pll_ofreq = 27000000;

		set_pll(btv);
		btwrite(0x11, BT848_TGCTRL);
		btwrite(0x41, BT848_DVSIF);
		
	} else {
		btv->pll.pll_ofreq = fsc;
		set_pll(btv);
		btwrite(0x0, BT848_DVSIF);
	}

	DBG("<-----tw_set_timing\n" ) ;
	
}

/* ----------------------------------------------------------------------- */


// ST_1201 for contrast/Sat control 
static void 
TW6801_contrast
(
	struct bttv *btv, 
	int cont
)
{

	DBG("TW6801: set contrast: %d\n", cont ); // DEBUG

	btv->contrast = cont ;

	btwrite( cont & 0xff, TW6801_CONTRAST ) ;
	
}



static void 
TW6801_sat
(struct bttv *btv, int color)
{
	int val_u, val_v ;

	DBG("TW6801: set saturation: %d\n", color ); // DEBUG

	btv->saturation = color;

	val_u   = color * 0x7F / 0x7F ;
	val_v   = color * 0x5A / 0x7F ;

	btwrite(val_u & 0xff, TW6801_SAT_U);

	btwrite(val_v & 0xff, TW6801_SAT_V);

}

static void 
TW6801_bright(struct bttv *btv, int bright)
{
	int value;

	DBG("TW6801: set bright: %d\n",bright); // DEBUG
	btv->bright = bright;

	// value = (bright >> 8) - 128;
	 value = bright  ;	
	
	btwrite(value & 0xff, TW6801_BRIGHT);
	
}

static void 
TW6801_hue(struct bttv *btv, int hue)
{
	int value;

	DBG("TW6801: set hue: %d\n",hue); // DEBUG
	
	btv->hue = hue;

	value = ( -128 * hue ) /90 ;
	
	if( value>127 ) {
		value = 127 ;
	} else if( value < -128 ) {
		value = -128 ;
	}
	
	btwrite(value & 0xff, TW6801_HUE);
	
}



static void bt848_bright(struct bttv *btv, int bright)
{
	int value;

	// printk("bttv: set bright: %d\n",bright); // DEBUG
	btv->bright = bright;

	/* We want -128 to 127 we get 0-65535 */
	value = (bright >> 8) - 128;
	btwrite(value & 0xff, BT848_BRIGHT);
}

static void bt848_hue(struct bttv *btv, int hue)
{
	int value;

	btv->hue = hue;

	/* -128 to 127 */
	value = (hue >> 8) - 128;
	btwrite(value & 0xff, BT848_HUE);
}



static void bt848_contrast(struct bttv *btv, int cont)
{
	int value,hibit;

	btv->contrast = cont;

	/* 0-511 */
	value = (cont  >> 7);
	hibit = (value >> 6) & 4;
	btwrite(value & 0xff, BT848_CONTRAST_LO);
	btaor(hibit, ~4, BT848_E_CONTROL);
	btaor(hibit, ~4, BT848_O_CONTROL);
	
}

static void bt848_sat(struct bttv *btv, int color)
{
	int val_u,val_v,hibits;

	// printk("TW6801: set saturation: %d\n",color); // DEBUG

	btv->saturation = color;

	/* 0-511 for the color */
	val_u   = ((color * btv->opt_uv_ratio) / 50) >> 7;
	val_v   = (((color * (100 - btv->opt_uv_ratio) / 50) >>7)*180L)/254;
	hibits  = (val_u >> 7) & 2;
	hibits |= (val_v >> 8) & 1;
	btwrite(val_u & 0xff, BT848_SAT_U_LO);
	btwrite(val_v & 0xff, BT848_SAT_V_LO);
	btaor(hibits, ~3, BT848_E_CONTROL);
	btaor(hibits, ~3, BT848_O_CONTROL);
}

/* ----------------------------------------------------------------------- */

static int
video_mux(struct bttv *btv, unsigned int input)
{
	int mux,mask2;

	if (input >= bttv_tvcards[btv->c.type].video_inputs)
		return -EINVAL;

	/* needed by RemoteVideo MX */
	mask2 = bttv_tvcards[btv->c.type].gpiomask2;
	
	if (mask2)
		gpio_inout(mask2,mask2);

	// ST_DEL_0924 for set video input
	/*
	if (input == btv->svhs)  {
		btor(BT848_CONTROL_COMP, BT848_E_CONTROL);
		btor(BT848_CONTROL_COMP, BT848_O_CONTROL);
	} else {
		btand(~BT848_CONTROL_COMP, BT848_E_CONTROL);
		btand(~BT848_CONTROL_COMP, BT848_O_CONTROL);
	}
	*/
	
	mux = bttv_tvcards[btv->c.type].muxsel[input] & 3;

	// ST_MODIFY_0924
	// btaor(mux<<5, ~(3<<5), BT848_IFORM);
	//dprintk(KERN_DEBUG "bttv%d: video mux: input=%d mux=%d\n",
	//	btv->c.nr,input,mux);

	// btaor( mux<<5, ~(3<<5), TW6801_IFORM ) ;	
	btaor( mux<<2, ~(TW6801_IFORM_YSEL), TW6801_IFORM ) ;		

	DBG( "TW6801-%d: video mux: input=%d mux=%d\n",
		btv->c.nr,input,mux ) ;

	/* card specific hook */
	if(bttv_tvcards[btv->c.type].muxsel_hook)
		bttv_tvcards[btv->c.type].muxsel_hook (btv, input);
	
	return 0;
	
}

static char *audio_modes[] = {
	"audio: tuner", "audio: radio", "audio: extern",
	"audio: intern", "audio: mute"
};

static int
audio_mux(struct bttv *btv, int input, int mute)
{
	int gpio_val, signal;
	struct v4l2_control ctrl;
	struct i2c_client *c;

	gpio_inout(bttv_tvcards[btv->c.type].gpiomask,
		   bttv_tvcards[btv->c.type].gpiomask);

	// ST_1115 for Audio Control 
	// signal = btread(BT848_DSTATUS) & BT848_DSTATUS_HLOC;
	signal = btread(TW6801_DSTATUS) & TW6801_DSTATUS_HLOCK ;

	btv->mute = mute;
	btv->audio = input;

	/* automute */
	mute = mute || (btv->opt_automute && !signal && !btv->radio_user);

	if (mute)
		gpio_val = bttv_tvcards[btv->c.type].gpiomute;
	else
		gpio_val = bttv_tvcards[btv->c.type].gpiomux[input];

	gpio_bits(bttv_tvcards[btv->c.type].gpiomask, gpio_val);
	
	if (bttv_gpio)
		tw_gpio_tracking(btv, audio_modes[mute ? 4 : input]);
	
	if (in_interrupt())
		return 0;

	ctrl.id = V4L2_CID_AUDIO_MUTE;
	ctrl.value = btv->mute;
	bttv_call_i2c_clients(btv, VIDIOC_S_CTRL, &ctrl);
	c = btv->i2c_msp34xx_client;
	if (c) {
		struct v4l2_routing route;

		/* Note: the inputs tuner/radio/extern/intern are translated
		   to msp routings. This assumes common behavior for all msp3400
		   based TV cards. When this assumption fails, then the
		   specific MSP routing must be added to the card table.
		   For now this is sufficient. */
		switch (input) {
		case TVAUDIO_INPUT_RADIO:
			route.input = MSP_INPUT(MSP_IN_SCART2, MSP_IN_TUNER1,
				    MSP_DSP_IN_SCART, MSP_DSP_IN_SCART);
			break;
		case TVAUDIO_INPUT_EXTERN:
			route.input = MSP_INPUT(MSP_IN_SCART1, MSP_IN_TUNER1,
				    MSP_DSP_IN_SCART, MSP_DSP_IN_SCART);
			break;
		case TVAUDIO_INPUT_INTERN:
			/* Yes, this is the same input as for RADIO. I doubt
			   if this is ever used. The only board with an INTERN
			   input is the BTTV_BOARD_AVERMEDIA98. I wonder how
			   that was tested. My guess is that the whole INTERN
			   input does not work. */
			route.input = MSP_INPUT(MSP_IN_SCART2, MSP_IN_TUNER1,
				    MSP_DSP_IN_SCART, MSP_DSP_IN_SCART);
			break;
		case TVAUDIO_INPUT_TUNER:
		default:
			route.input = MSP_INPUT_DEFAULT;
			break;
		}
		route.output = MSP_OUTPUT_DEFAULT;
		c->driver->command(c, VIDIOC_INT_S_AUDIO_ROUTING, &route);
	}
	c = btv->i2c_tvaudio_client;
	if (c) {
		struct v4l2_routing route;

		route.input = input;
		route.output = 0;
		c->driver->command(c, VIDIOC_INT_S_AUDIO_ROUTING, &route);
	}
	return 0;
}




static inline int
audio_mute(struct bttv *btv, int mute)
{
	return audio_mux(btv, btv->audio, mute);
}




static inline int
audio_input(struct bttv *btv, int input)
{
	return audio_mux(btv, input, btv->mute);
}




static void
i2c_vidiocschan(struct bttv *btv)
{
	v4l2_std_id std = bttv_tvnorms[btv->tvnorm].v4l2_id;

	bttv_call_i2c_clients(btv, VIDIOC_S_STD, &std);
	if (btv->c.type == BTTV_BOARD_VOODOOTV_FM)
		bttv_tda9880_setnorm(btv,btv->tvnorm);
}




static int
set_tvnorm
(
	struct bttv *btv, 
	unsigned int norm
)
{
	const struct bttv_tvnorm *tvnorm;

	DBG( "----->set_tvnorm\n" ) ;

	if (norm < 0 || norm >= BTTV_TVNORMS)
		return -EINVAL;

	btv->tvnorm = norm;
	tvnorm = &bttv_tvnorms[norm];

	// ST_DEL_0924
	/*
	btwrite(tvnorm->adelay, BT848_ADELAY);
	btwrite(tvnorm->bdelay, BT848_BDELAY);
	*/

	// ST_MODIFY_0924
	// btaor(tvnorm->iform,~(BT848_IFORM_NORM|BT848_IFORM_XTBOTH),
	//       BT848_IFORM);
	
	btaor( tvnorm->iform,~(TW6801_SDT_STD), TW6801_SDT ) ;

	// ST_DEL_0924
	// btwrite(tvnorm->vbipack, BT848_VBI_PACK_SIZE);
	// btwrite(1, BT848_VBI_PACK_DEL);
	
	btor( ( tvnorm->vbipack << TW6801_VBIC_DWC ) , TW6801_VBIC ) ;
	btor( (TW6801_VBIC_EVEN | TW6801_VBIC_ODD ),  TW6801_VBIC);

	// ST_DEl_1026
	// tw_set_timing(btv);

	switch (btv->c.type) {
	case BTTV_BOARD_VOODOOTV_FM:
		bttv_tda9880_setnorm(btv,norm);
		break;
	}
	
	DBG( "<-----set_tvnorm\n" ) ;	
	
	return 0;
	
}



static void
set_input
(
	struct bttv *btv, 
	unsigned int input
)
{
	unsigned long flags;

	btv->input = input;
	
	if (irq_iswitch) {
		
		spin_lock_irqsave(&btv->s_lock,flags);
		
		// ST_0126
		if (btv->curr.frame_irq) {
		// if (btv->curr.irqflags ) {			
			
			/* active capture -> delayed input switch */
			btv->new_input = input;
			
		} else {
		
			video_mux(btv,input);
			
		}
		
		spin_unlock_irqrestore(&btv->s_lock,flags);
		
	} else {

		// ST_NOTE_0924 for tw6801 support, set_video_input ....
		video_mux(btv,input);
		
	}

	// ST_1118 for Audio Control 
	/*
	audio_input(btv,(input == bttv_tvcards[btv->c.type].tuner ?
		       TVAUDIO_INPUT_TUNER : TVAUDIO_INPUT_EXTERN));
	*/
	audio_input(btv,(input == bttv_tvcards[btv->c.type].tuner ?
		       TW6801_AUDIO_SOURCE_TVTUNER : TW6801_AUDIO_SOURCE_EXT_AUDIO));
	
	set_tvnorm(btv,btv->tvnorm);
	
	i2c_vidiocschan(btv);

	
}



static void 
init_irqreg
(
	struct bttv *btv
)
{
	/* clear status */
	DBG( "----->init_irqreg \n" ) ;

	/* ST_0817 */
	/* ST_0129, for tw6805 * 8 */
	/* ST_0202 */
	// btwrite(	TW6801_VDMAPI | TW6801_DMAERR, TW6801_INT_MASK ) ; 
	btwrite(	TW6801_VDMAPI | TW6801_DMAERR | TW6801_FFOF, TW6801_INT_MASK ) ; 
	
	
	// btwrite(	TW6801_VDMAPI, TW6801_INT_MASK ) ;


	btv->m_isfirsttime = 0 ;

	DBG( "INT_MASK: %x\n ", btread( TW6801_INT_MASK ) ) ;

	DBG( "<-----init_irqreg \n" ) ;

}




static void 
init_tw680x
(
	struct bttv *btv
)
{
// 	int val;

	DBG( "----->init_tw680x\n" ) ;

	/* ST_0901 */
	btwrite(0x0000, 0x04C ) ;			// 0x04c


	btwrite(0x00, TW6801_DSTATUS ) ;	// 204
	btwrite(0x40, TW6801_IFORM ) ;		// 208 
	btwrite(0x04, TW6801_OPFORM ) ;		// 20C
	btwrite(0x00, TW6801_HSLEN ) ;		// 210
	btwrite(0x10, TW6801_ACNTL_1 ) ;		// 218		
	btwrite(0x02, TW6801_CROP_HI ) ;		// 21C, 02
	btwrite(0x18, TW6801_VDELAY_LO ) ;	// 220, 18	
	btwrite(0xF0, TW6801_VACTIVE_LO ) ;	// 224, f0	
	btwrite( 0x0F, TW6801_HDELAY_LO ) ;	// 228
	btwrite( 0xD0, TW6801_HACTIVE_LO ) ;	// 22C	
	btwrite(0x01, 0x3F0 ) ;					// 3F0
	btwrite(0x02, 0x3DC ) ;					// 3DC
	btwrite(0x1B, TW6801_VDELAY_LO2 ) ;					// 3E0	
	btwrite(0xF0, TW6801_VACTIVE_LO2 ) ;					// 3E4	
	btwrite(0x0F, 0x3E8 ) ;					// 3E8	
	btwrite(0xD0, 0x3EC ) ;					// 3EC	
	btwrite(0xCC, TW6801_CNTRL1 ) ;		// 230
	btwrite( 0x00, TW6801_VSCALE_LO ) ;	// 234
	btwrite( 0x11, TW6801_SCALE_HI ) ;		// 238
	btwrite( 0x00, TW6801_HSCALE_LO ) ; 	// 23C	
	btwrite( 0x00, 0x3F4 ) ;					// 3F4
	btwrite( 0x11, 0x3F8 ) ;					// 3F8
	btwrite( 0x00, 0x3FC ) ; 					// 3FC	
	btwrite(0x00, TW6801_BRIGHT ) ;		// 240
	btwrite(0x5C, TW6801_CONTRAST ) ;	// 244
	btwrite(0x98, TW6801_SHARPNESS_1 ) ;	// 248	
	btwrite(0x80, TW6801_SAT_U ) ;		// 24C
	btwrite(0x80, TW6801_SAT_V ) ;		// 250
	btwrite(0x00, TW6801_HUE ) ;		// 254
	btwrite(0xC6, TW6801_SHARPNESS_2 ) ;		// 258
	btwrite(0x84, TW6801_VSHARP ) ;		// 25C
	btwrite(0x44, TW6801_CORING ) ;		// 260
	btwrite(0x0A, TW6801_ACNTL_2 ) ;		// 268	0x00
	btwrite(0x00, TW6801_SDT ) ;			// 270	
	btwrite(0x7F, TW6801_SDTR ) ;		// 274
	btwrite(0x07, 0x278 ) ;				// 278
	btwrite(0x7F, 0x27C ) ;				// 27C
	btwrite(0x50, TW6801_CLMPG ) ;		// 280
	btwrite(0x42, TW6801_IAGC ) ;		// 284
	btwrite(0xF0, TW6801_AGCGAIN ) ;	// 288	
	btwrite(0xD8, TW6801_PEAKWT ) ;		// 28C
	btwrite(0xBC, TW6801_CLMPL ) ;		// 290
	btwrite(0xB8, TW6801_SYNCT ) ;		// 294
	btwrite(0x44, TW6801_MISSCNT ) ;	// 298
	btwrite(0x2A, TW6801_PCLAMP ) ;		// 29C	
	btwrite(0x0C, TW6801_VCNTL_1 ) ;		// 2A0
	btwrite(0x00, TW6801_VCNTL_2 ) ;		// 2A4
	btwrite(0x78, TW6801_CKILL ) ;		// 2A8
	btwrite(0x44, TW6801_COMB ) ;		// 2AC
	btwrite(0xB0, TW6801_LDLY ) ;		// 2B0		
	btwrite(0x14, TW6801_MISC_1 ) ;		// 2B4
	btwrite(0x3D, TW6801_LOOP ) ;		// 2B8	
	btwrite(0x06, TW6801_MISC_2 ) ;		// 2BC
	btwrite(0x00, TW6801_MVSN ) ;		// 2C0
	btwrite(0x00, TW6801_STATUS_2 ) ;	// 2C4	
	btwrite(0xA0, TW6801_HFREF ) ;		// 2C8
	btwrite( 0x84, TW6801_CLMD ) ;		// 2CC
	btwrite(0x00, TW6801_IDCNTL ) ;		// 2D0
	btwrite(0x80, TW6801_CLCNTL_1 ) ;	// 2D4	
	btwrite(0x00, 0x2D8 ) ;				// 2D8
	btwrite(0x03, TW6801_VBIC );			// 0x010
	btwrite(0x43, TW6801_CAP_CTL);		// 0x40, 43

	/* ST_0129 */
	// btwrite( 0x0110, TW68XX_SET_DMATRIG );	// 0x060		
	// btwrite( 0x01080, TW6801_VDMAC );	// 0x00		
	//btwrite( 0x0120, TW68XX_SET_DMATRIG );	// 0x060		
	//btwrite( 0x00180, TW6801_VDMAC );	// 0x00		
	btwrite( 0x0120, TW68XX_SET_DMATRIG );	// 0x060		
	btwrite( 0x00180, TW6801_VDMAC );	// 0x00		



	btwrite( 0x00, TW6801_TEST_REG  );	// 0x2C	
	TW6801_contrast(btv, btv->contrast);
	TW6801_sat(btv,      btv->saturation);
	TW6801_bright(btv,   btv->bright);
	TW6801_hue(btv,      btv->hue);
	
	DBG( "<-----init_tw680x\n" ) ;


}



static void 
bttv_reinit_tw680x
(
	struct bttv *btv
)
{
	unsigned long flags;

	DBG_SWITCH( "----->bttv_reinit_tw680x\n" ) ;

	if (bttv_verbose)
		DBG(KERN_INFO "bttv%d: reset, reinitialize\n",btv->c.nr);

	// ST_11202008
	init_tw680x(btv);
	init_irqreg(btv);
	//
	
	spin_lock_irqsave(&btv->s_lock,flags);
	btv->errors=0;
	
	tw_DisableDMA( btv ) ;

	spin_unlock_irqrestore(&btv->s_lock,flags);
	
	btor( (TW6801_VDMAC_VDMAP_EN|TW6801_VDMAC_VFIFO_EN), TW6801_VDMAC );	

	btwrite( btv->main.dma, TW6801_VDMAP_SA ) ;
	
	btv->pll.pll_current = -1;
	
	set_input(btv,btv->input);
	
	DBG( "<-----bttv_reinit_tw680x\n" ) ;
	

}



static int 
get_control
(
	struct bttv *btv, 
	struct v4l2_control *c
)
{
	struct video_audio va;
	int i;

	for (i = 0; i < BTTV_CTLS; i++)
		if (bttv_ctls[i].id == c->id)
			break;
	if (i == BTTV_CTLS)
		return -EINVAL;
	if (btv->audio_hook && i >= 4 && i <= 8) {
		memset(&va,0,sizeof(va));
		btv->audio_hook(btv,&va,0);
		switch (c->id) {
		case V4L2_CID_AUDIO_MUTE:
			c->value = (VIDEO_AUDIO_MUTE & va.flags) ? 1 : 0;
			break;
		case V4L2_CID_AUDIO_VOLUME:
			c->value = va.volume;
			break;
		case V4L2_CID_AUDIO_BALANCE:
			c->value = va.balance;
			break;
		case V4L2_CID_AUDIO_BASS:
			c->value = va.bass;
			break;
		case V4L2_CID_AUDIO_TREBLE:
			c->value = va.treble;
			break;
		}
		return 0;
	}
	switch (c->id) {
	case V4L2_CID_BRIGHTNESS:
		c->value = btv->bright;
		break;
	case V4L2_CID_HUE:
		c->value = btv->hue;
		break;
	case V4L2_CID_CONTRAST:
		c->value = btv->contrast;
		break;
	case V4L2_CID_SATURATION:
		c->value = btv->saturation;
		break;

	case V4L2_CID_AUDIO_MUTE:
	case V4L2_CID_AUDIO_VOLUME:
	case V4L2_CID_AUDIO_BALANCE:
	case V4L2_CID_AUDIO_BASS:
	case V4L2_CID_AUDIO_TREBLE:
		bttv_call_i2c_clients(btv,VIDIOC_G_CTRL,c);
		break;

	case V4L2_CID_PRIVATE_CHROMA_AGC:
		c->value = btv->opt_chroma_agc;
		break;
	case V4L2_CID_PRIVATE_COMBFILTER:
		c->value = btv->opt_combfilter;
		break;
	case V4L2_CID_PRIVATE_LUMAFILTER:
		c->value = btv->opt_lumafilter;
		break;
	case V4L2_CID_PRIVATE_AUTOMUTE:
		c->value = btv->opt_automute;
		break;
	case V4L2_CID_PRIVATE_AGC_CRUSH:
		c->value = btv->opt_adc_crush;
		break;
	case V4L2_CID_PRIVATE_VCR_HACK:
		c->value = btv->opt_vcr_hack;
		break;
	case V4L2_CID_PRIVATE_WHITECRUSH_UPPER:
		c->value = btv->opt_whitecrush_upper;
		break;
	case V4L2_CID_PRIVATE_WHITECRUSH_LOWER:
		c->value = btv->opt_whitecrush_lower;
		break;
	case V4L2_CID_PRIVATE_UV_RATIO:
		c->value = btv->opt_uv_ratio;
		break;
	case V4L2_CID_PRIVATE_FULL_LUMA_RANGE:
		c->value = btv->opt_full_luma_range;
		break;
	case V4L2_CID_PRIVATE_CORING:
		c->value = btv->opt_coring;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}


static int 
set_control
(
	struct bttv *btv, 
	struct v4l2_control *c
)
{
	struct video_audio va;
	int i,val;

	for (i = 0; i < BTTV_CTLS; i++)
		if (bttv_ctls[i].id == c->id)
			break;
	if (i == BTTV_CTLS)
		return -EINVAL;
	if (btv->audio_hook && i >= 4 && i <= 8) {
		memset(&va,0,sizeof(va));
		btv->audio_hook(btv,&va,0);
		switch (c->id) {
		case V4L2_CID_AUDIO_MUTE:
			if (c->value) {
				va.flags |= VIDEO_AUDIO_MUTE;
				audio_mute(btv, 1);
			} else {
				va.flags &= ~VIDEO_AUDIO_MUTE;
				audio_mute(btv, 0);
			}
			break;

		case V4L2_CID_AUDIO_VOLUME:
			va.volume = c->value;
			break;
		case V4L2_CID_AUDIO_BALANCE:
			va.balance = c->value;
			break;
		case V4L2_CID_AUDIO_BASS:
			va.bass = c->value;
			break;
		case V4L2_CID_AUDIO_TREBLE:
			va.treble = c->value;
			break;
		}
		btv->audio_hook(btv,&va,1);
		return 0;
	}


	// printk( "-->Set_Control->CID: %x\n", c->id ) ;
	
	switch (c->id) {
	case V4L2_CID_BRIGHTNESS:
		
		// bt848_bright(btv,c->value);
		TW6801_bright(btv,c->value);		
		
		break;
	case V4L2_CID_HUE:

		// ST_1201
		// bt848_hue(btv,c->value);
		TW6801_hue( btv, c->value ) ;
		
		break;
	case V4L2_CID_CONTRAST:
		
		// bt848_contrast(btv,c->value);
		TW6801_contrast(btv,c->value);		

		break;
	case V4L2_CID_SATURATION:
		// bt848_sat(btv,c->value);
		TW6801_sat(btv,c->value);		
		break;
	case V4L2_CID_AUDIO_MUTE:
		audio_mute(btv, c->value);
		/* fall through */
	case V4L2_CID_AUDIO_VOLUME:
	case V4L2_CID_AUDIO_BALANCE:
	case V4L2_CID_AUDIO_BASS:
	case V4L2_CID_AUDIO_TREBLE:
		bttv_call_i2c_clients(btv,VIDIOC_S_CTRL,c);
		break;

	case V4L2_CID_PRIVATE_CHROMA_AGC:
		btv->opt_chroma_agc = c->value;
		val = btv->opt_chroma_agc ? BT848_SCLOOP_CAGC : 0;
		btwrite(val, BT848_E_SCLOOP);
		btwrite(val, BT848_O_SCLOOP);
		break;
	case V4L2_CID_PRIVATE_COMBFILTER:
		btv->opt_combfilter = c->value;
		break;
	case V4L2_CID_PRIVATE_LUMAFILTER:
		btv->opt_lumafilter = c->value;
		if (btv->opt_lumafilter) {
			btand(~BT848_CONTROL_LDEC, BT848_E_CONTROL);
			btand(~BT848_CONTROL_LDEC, BT848_O_CONTROL);
		} else {
			btor(BT848_CONTROL_LDEC, BT848_E_CONTROL);
			btor(BT848_CONTROL_LDEC, BT848_O_CONTROL);
		}
		break;
	case V4L2_CID_PRIVATE_AUTOMUTE:
		btv->opt_automute = c->value;
		break;
	case V4L2_CID_PRIVATE_AGC_CRUSH:
		btv->opt_adc_crush = c->value;
		btwrite(BT848_ADC_RESERVED | (btv->opt_adc_crush ? BT848_ADC_CRUSH : 0),
			BT848_ADC);
		break;
	case V4L2_CID_PRIVATE_VCR_HACK:
		btv->opt_vcr_hack = c->value;
		break;
	case V4L2_CID_PRIVATE_WHITECRUSH_UPPER:
		btv->opt_whitecrush_upper = c->value;
		btwrite(c->value, BT848_WC_UP);
		break;
	case V4L2_CID_PRIVATE_WHITECRUSH_LOWER:
		btv->opt_whitecrush_lower = c->value;
		btwrite(c->value, BT848_WC_DOWN);
		break;
	case V4L2_CID_PRIVATE_UV_RATIO:
		btv->opt_uv_ratio = c->value;
		
		// bt848_sat(btv, btv->saturation);
		TW6801_sat(btv, btv->saturation);		
		
		break;
	case V4L2_CID_PRIVATE_FULL_LUMA_RANGE:
		btv->opt_full_luma_range = c->value;
		btaor((c->value<<7), ~BT848_OFORM_RANGE, BT848_OFORM);
		break;
	case V4L2_CID_PRIVATE_CORING:
		btv->opt_coring = c->value;
		btaor((c->value<<5), ~BT848_OFORM_CORE32, BT848_OFORM);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/* ----------------------------------------------------------------------- */


void tw_gpio_tracking(struct bttv *btv, char *comment)
{
	unsigned int outbits, data;

	// ST_MODIFY_0923 for TW6801 support 
	// outbits = btread(BT848_GPIO_OUT_EN);
	// data    = btread(BT848_GPIO_DATA);

	outbits = btread(TW6801_GPOE) ;
	data    = btread(TW6801_GPDATA) ;
	
	DBG(KERN_DEBUG "TW6801-%d: gpio: en=%08x, out=%08x in=%08x [%s]\n",
	       btv->c.nr,outbits,data & outbits, data & ~outbits, comment);
	
}

static void bttv_field_count(struct bttv *btv)
{
	int need_count = 0;

	DBG( "----->bttv_field_count\n" ) ;

	if (btv->users)
		need_count++;

	if (need_count) {
		/* start field counter */
		// ST_0725
		// btor(BT848_INT_VSYNC,BT848_INT_MASK);
		// ST_DEL_1122
		// btor( TW6801_VLOCK, TW6801_INT_MASK);

	} else {
		/* stop field counter */
		// ST_0725	
		// btand(~BT848_INT_VSYNC,BT848_INT_MASK);
		// ST_DEL_1122
		//btand(~TW6801_VLOCK, TW6801_INT_MASK );

		btv->field_count = 0;
	}

	DBG( "<-----bttv_field_count\n" ) ;
	
}

static const struct bttv_format*
format_by_palette(int palette)
{
	unsigned int i;

	for (i = 0; i < BTTV_FORMATS; i++) {
		if (-1 == bttv_formats[i].palette)
			continue;
		if (bttv_formats[i].palette == palette)
			return bttv_formats+i;
	}
	return NULL;
}

static const struct bttv_format*
format_by_fourcc(int fourcc)
{
	unsigned int i;

	for (i = 0; i < BTTV_FORMATS; i++) {
		if (-1 == bttv_formats[i].fourcc)
			continue;
		if (bttv_formats[i].fourcc == fourcc)
			return bttv_formats+i;
	}
	return NULL;
}

/* ----------------------------------------------------------------------- */
/* misc helpers                                                            */

static int
bttv_switch_overlay(struct bttv *btv, struct bttv_fh *fh,
		    struct bttv_buffer *new)
{
	struct bttv_buffer *old;
	unsigned long flags;
	int retval = 0;

	DBG_SWITCH("switch_overlay: enter [new=%p]\n",new);


	if (new)
		new->vb.state = STATE_DONE;

	spin_lock_irqsave(&btv->s_lock,flags);

	old = btv->screen;

	btv->screen = new;

	spin_unlock_irqrestore(&btv->s_lock,flags);
	if (NULL == new)
		free_btres(btv,fh,RESOURCE_OVERLAY);
	if (NULL != old) {
		dprintk("switch_overlay: old=%p state is %d\n",old,old->vb.state);
		tw_dma_free(&fh->cap,btv, old);
		kfree(old);
	}

	DBG("switch_overlay: done\n");

	return retval;
}

/* ----------------------------------------------------------------------- */
/* video4linux (1) interface                                               */

static int 
tw_prepare_buffer
(
	struct videobuf_queue *q,
	struct bttv *btv,
	struct bttv_buffer *buf,
	const struct bttv_format *fmt,
	unsigned int width, 
	unsigned int height,
	enum v4l2_field field
)
{
	int redo_dma_risc = 0;
	int rc;

	// DBG_SWITCH( "----->tw_prepare_buffer\n" ) ;

	/* check settings */
	if (NULL == fmt) {
		DBG_MMAP( "<-----tw_prepare_buffer!!\n" ) ;
		return -EINVAL;
	}

	// ST_DEL_1001
	DBG_SWITCH( "----->tw_prepare_buffer-%d :: W: %d, H: %d,SW: %d,SH: %d, Field: %d ...\n", 
		btv->c.nr,
		width, height,
		bttv_tvnorms[btv->tvnorm].swidth,
		bttv_tvnorms[btv->tvnorm].sheight,
		field ) ;
	
	if (fmt->btformat == BT848_COLOR_FMT_RAW) {		
		width  = RAW_BPL;
		height = RAW_LINES*2;
		if (width*height > buf->vb.bsize)
			return -EINVAL;
		buf->vb.size = buf->vb.bsize;
	} else {
		if (width  < 48 ||
		    height < 32 ||
		    width  > bttv_tvnorms[btv->tvnorm].swidth ||
		    height > bttv_tvnorms[btv->tvnorm].sheight) {
			return -EINVAL;
		}
		
		buf->vb.size = (width * height * fmt->depth) >> 3;

		if (0 != buf->vb.baddr  &&  buf->vb.bsize < buf->vb.size){
			DBG_MMAP( "bsize < size .......\n" ) ;
			return -EINVAL;
		}	

	}


	/* alloc + fill struct bttv_buffer (if changed) */
	if (	buf->vb.width != width || 
		buf->vb.height != height ||
	    	buf->vb.field != field ||
	    	buf->tvnorm != btv->tvnorm || 
	    	buf->fmt != fmt) {
	    	
		buf->vb.width  	= width;
		buf->vb.height 	= height ;
		buf->vb.field  		= field;
		buf->tvnorm    	= btv->tvnorm ;
		buf->fmt       		= fmt;
		redo_dma_risc 		= 1 ;
		
	}


	/* alloc risc memory */

	if ( STATE_NEEDS_INIT == buf->vb.state ) {

		DBG_MMAP( "STATE_NEEDS_INIT...\n" ) ;
		
		redo_dma_risc = 1 ;
		
		if (0 != (rc = videobuf_iolock(q,&buf->vb,&btv->fbuf)))
			goto fail ;
		
	}

	/* ST_0727 */
	// redo_dma_risc = 1 ;


	if ( redo_dma_risc ) {

		DBG_MMAP( "-->redo_dma_risc - %d\n", btv->c.nr ) ;
		
		if (0 != (rc = tw_buffer_risc(btv,buf)))
			goto fail;

	}

	buf->vb.state = STATE_PREPARED;

	DBG_SWITCH( "<-----tw_prepare_buffer\n" ) ;
	
	return 0;

 fail:

	DBG_FPS( "-->redo_dma_risc-Failed - %d ... \n", btv->c.nr ) ;	
	
	tw_dma_free(q,btv,buf);
	
	return rc;

	
}



static int
buffer_setup
(
	struct videobuf_queue *q, 
	unsigned int *count, 
	unsigned int *size
)
{
	struct bttv_fh *fh = q->priv_data;

	*size = fh->fmt->depth*fh->width*fh->height >> 3;

	if (0 == *count)
		*count = gbuffers;
	
	while (*size * *count > gbuffers * gbufsize)
		(*count)--;
	
	return 0;
	
}



static int
buffer_prepare
(
	struct videobuf_queue *q, 
	struct videobuf_buffer *vb,
      	enum v4l2_field field
)
{
	struct bttv_buffer *buf = container_of(vb,struct bttv_buffer,vb);
	struct bttv_fh *fh = q->priv_data ;

	DBG("buffer_prepare-%d: %d \n", fh->btv->c.nr, field ) ;

	return tw_prepare_buffer(	q,
							fh->btv, 
							buf, 
							fh->fmt,
				   			fh->width, 
				   			fh->height, 
				   			field ) ;
	
}



static void
buffer_queue(struct videobuf_queue *q, struct videobuf_buffer *vb)
{
	struct bttv_buffer *buf = container_of(vb,struct bttv_buffer,vb) ;
	struct bttv_fh *fh = q->priv_data ;
	struct bttv    *btv = fh->btv ;

	unsigned long flags ;
	struct timeval ts ;

	do_gettimeofday(&ts);

	DBG( "----->buffer_queue-%d, ts: %d \n", btv->c.nr, ( (ts.tv_sec * 1000) + (ts.tv_usec/1000)) ) ;

	buf->vb.state = STATE_QUEUED ;

	//list_add_tail( new, target ) ==> Add new to head of target...
	list_add_tail(&buf->vb.queue,&btv->capture) ;

	// tw_EnableDMA( fh->btv ) ;

	DBG( "<-----buffer_queue\n" ) ;
	
}



static void buffer_release(struct videobuf_queue *q, struct videobuf_buffer *vb)
{
	struct bttv_buffer *buf = container_of(vb,struct bttv_buffer,vb);
	struct bttv_fh *fh = q->priv_data;

	// ST_0617, Try to fix STREAMOFF with device die issue ... 
	struct bttv *btv = (struct bttv *)fh->btv ;
	struct bttv_buffer *item;

	DBG_FPS( "--->buffer_release \n" ) ;
	

	// ST_0618_ADD, try to fix streamoff with device die issue ... 
#if 0
	//DBG_FPS ("---->buffer_release-%d\n", btv->c.nr ) ;

	while (!list_empty(&btv->capture)) {
		item = list_entry(btv->capture.next, struct bttv_buffer, vb.queue);
		list_del(&item->vb.queue);
		item->vb.state = STATE_ERROR;
		wake_up(&item->vb.done);
		tw_dma_free( q, fh->btv, item )  ;	
		//DBG_FPS ("== free dma-%d :: %x == \n", btv->c.nr, item ) ;
	}

	//DBG_FPS ("<----buffer_release-%d\n", btv->c.nr ) ;


#else

	// ST_0617_TEST
	tw_dma_free( q, fh->btv,buf);

#endif

}

static struct videobuf_queue_ops bttv_video_qops = {
	.buf_setup    = buffer_setup,
	.buf_prepare  = buffer_prepare,
	.buf_queue    = buffer_queue,
	.buf_release  = buffer_release,
};



static int 
tw_common_ioctls
(
	struct bttv *btv, 
	unsigned int cmd, 
	void *arg
)
{

	DBG( "----->tw_common_ioctls \n" ) ;

	switch (cmd) {
	case BTTV_VERSION:
		// ST_MODIFY_0924 for tw6801 support 
		// return BTTV_VERSION_CODE;
		return TW6801_VERSION_CODE;		

	/* ***  v4l1  *** ************************************************ */
	case VIDIOCGFREQ:
	{
		unsigned long *freq = arg;

		DBG( "**VIDIOCGFREQ**\n" ) ;	
		
		*freq = btv->freq;
		return 0;
	}
	case VIDIOCSFREQ:
	{
		struct v4l2_frequency freq;
		
		DBG( "**(tw_common_ioctls)VIDIOCSFREQ**\n" ) ;	
		
		memset(&freq, 0, sizeof(freq));
		freq.frequency = *(unsigned long *)arg;
		mutex_lock(&btv->lock);
		freq.type = btv->radio_user ? V4L2_TUNER_RADIO : V4L2_TUNER_ANALOG_TV;
		btv->freq = *(unsigned long *)arg;

		bttv_call_i2c_clients(btv,VIDIOC_S_FREQUENCY,&freq);
		
		if (btv->has_matchbox && btv->radio_user) {
			tea5757_set_freq(btv,*(unsigned long *)arg);
		}
		
		mutex_unlock(&btv->lock);
		
		return 0;
	}

	case VIDIOCGTUNER:
	{
		struct video_tuner *v = arg;

		DBG( "\n===== VIDIOCGTUNER =====\n" ) ;	

		if (UNSET == bttv_tvcards[btv->c.type].tuner) {
			DBG( "----->Tuner, UNSET\n" ) ;
			return -EINVAL;
		}
		
		if (v->tuner) /* Only tuner 0 */{
			DBG( "----->Tuner, Only tuner 0\n" ) ;
			return -EINVAL;
		}
		
		strcpy(v->name, "Television") ;
		DBG( "----->Television ..... \n" ) ;
		
		v->rangelow  = 0 ;
		v->rangehigh = 0x7FFFFFFF ;
		v->flags     = VIDEO_TUNER_PAL|VIDEO_TUNER_NTSC |VIDEO_TUNER_SECAM ;

		// ST_TEST_0318 for VBI support 
		// v->mode      = btv->tvnorm ;
		v->mode      = 1 ; 		// ZVBI library ==> 0: PAL, 1:NTSC, 2:SECM
	
		// ST_1117
		// v->signal    = (btread(BT848_DSTATUS)&BT848_DSTATUS_HLOC) ? 0xFFFF : 0;
		v->signal    = (btread(TW6801_DSTATUS)&TW6801_DSTATUS_HLOCK) ? 0xFFFF : 0 ;
		
		bttv_call_i2c_clients(btv,cmd,v) ;
		
		return 0 ;
		
	}
	case VIDIOCSTUNER:
	{
		struct video_tuner *v = arg;

		DBG( "===== VIDIOCSTUNER ===== \n" ) ;	
		
		if (v->tuner) /* Only tuner 0 */
			return -EINVAL;
		if (v->mode >= BTTV_TVNORMS)
			return -EINVAL;

		mutex_lock(&btv->lock);
		set_tvnorm(btv,v->mode);
		bttv_call_i2c_clients(btv,cmd,v);
		mutex_unlock(&btv->lock);
		return 0;
	}

	case VIDIOCGCHAN:
	{
		struct video_channel *v = arg;
		unsigned int channel = v->channel;


		DBG( "**VIDIOCGCHAN**\n" ) ;	

		if (channel >= bttv_tvcards[btv->c.type].video_inputs)
			return -EINVAL;
		v->tuners=0;
		v->flags = VIDEO_VC_AUDIO;
		v->type = VIDEO_TYPE_CAMERA;
		v->norm = btv->tvnorm;
		if (channel == bttv_tvcards[btv->c.type].tuner)  {
			strcpy(v->name,"Television");
			v->flags|=VIDEO_VC_TUNER;
			v->type=VIDEO_TYPE_TV;
			v->tuners=1;
		} else if (channel == btv->svhs) {
			strcpy(v->name,"S-Video");
		} else {
			sprintf(v->name,"Composite%d",channel);
		}
		return 0;
	}
	case VIDIOCSCHAN:
	{
		struct video_channel *v = arg;
		unsigned int channel = v->channel;

		DBG( "**VIDIOCSCHAN**\n" ) ;	
		DBG( "Channel==> %d\n", v->channel ) ;

		if (channel >= bttv_tvcards[btv->c.type].video_inputs)
			return -EINVAL;
		if (v->norm >= BTTV_TVNORMS)
			return -EINVAL;

		mutex_lock(&btv->lock);
		if (channel == btv->input &&
		    v->norm == btv->tvnorm) {
			/* nothing to do */
			mutex_unlock(&btv->lock);
			return 0;
		}

		btv->tvnorm = v->norm;
		set_input(btv,v->channel);
		mutex_unlock(&btv->lock);
		return 0;
	}

	case VIDIOCGAUDIO:
	{
		struct video_audio *v = arg;


		DBG( "**VIDIOCGAUDIO**\n" ) ;	

		memset(v,0,sizeof(*v));
		strcpy(v->name,"Television");
		v->flags |= VIDEO_AUDIO_MUTABLE;
		v->mode  = VIDEO_SOUND_MONO;

		mutex_lock(&btv->lock);
		bttv_call_i2c_clients(btv,cmd,v);

		/* card specific hooks */
		if (btv->audio_hook)
			btv->audio_hook(btv,v,0);

		mutex_unlock(&btv->lock);
		return 0;
	}
	case VIDIOCSAUDIO:
	{
		struct video_audio *v = arg;
		unsigned int audio = v->audio;

		
		DBG( "**VIDIOCSAUDIO**\n" ) ;	

		if (audio >= bttv_tvcards[btv->c.type].audio_inputs)
			return -EINVAL;

		mutex_lock(&btv->lock);
		audio_mute(btv, (v->flags&VIDEO_AUDIO_MUTE) ? 1 : 0);
		bttv_call_i2c_clients(btv,cmd,v);

		/* card specific hooks */
		if (btv->audio_hook)
			btv->audio_hook(btv,v,1);

		mutex_unlock(&btv->lock);
		return 0;
	}

	/* ***  v4l2  *** ************************************************ */
	case VIDIOC_ENUMSTD:
	{
		struct v4l2_standard *e = arg;
		unsigned int index = e->index;


		DBG( "**VIDIOC_ENUMSTD**\n" ) ;	

		if (index >= BTTV_TVNORMS) {
			DBG( "<-----tw_common_ioctls \n" ) ;				
			return -EINVAL;
		}

		DBG( "Index: %d \n", index ) ;
		
		v4l2_video_std_construct(e, bttv_tvnorms[e->index].v4l2_id,
					 bttv_tvnorms[e->index].name);
		
		e->index = index;
		
		return 0;
	}
	case VIDIOC_G_STD:
	{
		v4l2_std_id *id = arg;

		DBG( "**VIDIOC_G_STD**\n" ) ;	
		
		*id = bttv_tvnorms[btv->tvnorm].v4l2_id;
		return 0;
	}
	case VIDIOC_S_STD:
	{
		v4l2_std_id *id = arg;
		unsigned int i;


		DBG( "**VIDIOC_S_STD**\n" ) ;	
		
		for (i = 0; i < BTTV_TVNORMS; i++)
			if (*id & bttv_tvnorms[i].v4l2_id)
				break;
		if (i == BTTV_TVNORMS)
			return -EINVAL;

		mutex_lock(&btv->lock);
		set_tvnorm(btv,i);
		i2c_vidiocschan(btv);
		mutex_unlock(&btv->lock);
		return 0;
	}
	case VIDIOC_QUERYSTD:
	{
		v4l2_std_id *id = arg;


		DBG( "===== VIDIOC_QUERYSTD ===== \n" ) ;	

		// ST_TEST_0318 for VBI support
		/*
		if (btread(BT848_DSTATUS) & BT848_DSTATUS_NUML)
			*id = V4L2_STD_625_50;
		else
			*id = V4L2_STD_525_60;
		*/
		if (btread( TW6801_DSTATUS) & TW6801_DSTATUS_DET50 ) {
			DBG( "50 Hz detected .../n" ) ;
			*id = V4L2_STD_625_50;
		} else {
			DBG( "60 Hz detected .../n" ) ;
			*id = V4L2_STD_525_60;
		}

		return 0;
	}

	case VIDIOC_ENUMINPUT:
	{
		struct v4l2_input *i = arg;
		unsigned int n;

		DBG( "**VIDIOC_ENUMINPUT**\n" ) ;

		n = i->index;
		
		if (n >= bttv_tvcards[btv->c.type].video_inputs)
			return -EINVAL;
		
		memset(i,0,sizeof(*i));
		
		i->index    = n;

		i->type     = V4L2_INPUT_TYPE_CAMERA;

		i->audioset = 0;

		if (i->index == bttv_tvcards[btv->c.type].tuner) {
			sprintf(i->name, "Television");
			i->type  = V4L2_INPUT_TYPE_TUNER;
			i->tuner = 0;
		} else if (i->index == btv->svhs) {
			sprintf(i->name, "S-Video");
		} else {
			sprintf(i->name,"Composite%d",i->index);
		}

		/* ST_0904, make sure the status is "0" */
		i->status = 0 ;
		
		if (i->index == btv->input) {
	
			__u32 dstatus = btread(TW6801_DSTATUS);

			// Video signal loss
			if (0x80 == (dstatus & TW6801_DSTATUS_VDLOSS))
				i->status |= V4L2_IN_ST_NO_SIGNAL;

			// Horizontal sync PLL lock ?
			if (0 == (dstatus & TW6801_DSTATUS_HLOCK))
				i->status |= V4L2_IN_ST_NO_H_LOCK;
			
		}


		DBG( "--->dev-%d, name: %s, index: %d, DSTATUS: %x \n", btv->c.nr,  i->name, i->index, i->status ) ;
		
		for (n = 0; n < BTTV_TVNORMS; n++)
			i->std |= bttv_tvnorms[n].v4l2_id;
		
		return 0;
	}
	case VIDIOC_G_INPUT:
	{
		int *i = arg;

		DBG( "**VIDIOC_G_INPUT**\n" ) ;	
		
		*i = btv->input ;
		
		return 0 ;
	}
	case VIDIOC_S_INPUT:
	{
		unsigned int *i = arg;

		DBG( "**VIDIOC_S_INPUT** ( Input: %d ) \n", *i ) ;	

		if (*i > bttv_tvcards[btv->c.type].video_inputs) {
			DBG( "**VIDIOC_S_INPUT got error ...**\n" ) ;
			return -EINVAL;
		}
		
		mutex_lock(&btv->lock);
		
		set_input(btv,*i);
		
		mutex_unlock(&btv->lock);
		
		return 0;
		
	}

	case VIDIOC_G_TUNER:
	{
		struct v4l2_tuner *t = arg;


		DBG( "**VIDIOC_G_TUNER**\n" ) ;			

		if (UNSET == bttv_tvcards[btv->c.type].tuner) {
			DBG( "==> Tuner is UNSET ... \n" ) ;						
			return -EINVAL;
		}			
		
		if (0 != t->index) {
			DBG( "==> Index is 0 ... \n" ) ;
			return -EINVAL;
		}
		
		mutex_lock(&btv->lock);
		
		memset(t,0,sizeof(*t));

		t->rxsubchans = V4L2_TUNER_SUB_MONO;

		bttv_call_i2c_clients(btv, VIDIOC_G_TUNER, t);

		strcpy(t->name, "Television");

		t->capability = V4L2_TUNER_CAP_NORM;
		
		t->type       = V4L2_TUNER_ANALOG_TV;

		// ST_MODIFY for get tuner
		// if (btread(BT848_DSTATUS)&BT848_DSTATUS_HLOC)
		
		if (btread(TW6801_DSTATUS)&TW6801_DSTATUS_HLOCK) {
			// DBG( "==> Sgnal is 0xFFFF( No Signal in there ) ... \n" ) ;
			t->signal = 0xffff;
		}

		if (btv->audio_hook) {
			/* Hmmm ... */
			struct video_audio va;
			memset(&va, 0, sizeof(struct video_audio));
			btv->audio_hook(btv,&va,0);
			t->audmode    = V4L2_TUNER_MODE_MONO;
			t->rxsubchans = V4L2_TUNER_SUB_MONO;
			if(va.mode & VIDEO_SOUND_STEREO) {
				t->audmode    = V4L2_TUNER_MODE_STEREO;
				t->rxsubchans = V4L2_TUNER_SUB_STEREO;
			}
			if(va.mode & VIDEO_SOUND_LANG2) {
				t->audmode    = V4L2_TUNER_MODE_LANG1;
				t->rxsubchans = V4L2_TUNER_SUB_LANG1
					| V4L2_TUNER_SUB_LANG2;
			}
		}
		
		/* FIXME: fill capability+audmode */
		
		mutex_unlock(&btv->lock);
		
		return 0;
	}
	case VIDIOC_S_TUNER:
	{
		struct v4l2_tuner *t = arg;


		DBG( "**VIDIOC_S_TUNER**\n" ) ;	

		if (UNSET == bttv_tvcards[btv->c.type].tuner)
			return -EINVAL;
		
		if (0 != t->index)
			return -EINVAL;
		mutex_lock(&btv->lock);
		bttv_call_i2c_clients(btv, VIDIOC_S_TUNER, t);
		if (btv->audio_hook) {
			struct video_audio va;
			memset(&va, 0, sizeof(struct video_audio));
			if (t->audmode == V4L2_TUNER_MODE_MONO)
				va.mode = VIDEO_SOUND_MONO;
			else if (t->audmode == V4L2_TUNER_MODE_STEREO ||
				 t->audmode == V4L2_TUNER_MODE_LANG1_LANG2)
				va.mode = VIDEO_SOUND_STEREO;
			else if (t->audmode == V4L2_TUNER_MODE_LANG1)
				va.mode = VIDEO_SOUND_LANG1;
			else if (t->audmode == V4L2_TUNER_MODE_LANG2)
				va.mode = VIDEO_SOUND_LANG2;
			btv->audio_hook(btv,&va,1);
		}
		mutex_unlock(&btv->lock);
		return 0;
	}

	case VIDIOC_G_FREQUENCY:
	{
		struct v4l2_frequency *f = arg;


		//printk( "(tw_common_ioctls)----->VIDIOC_G_FREQUENCY\n" ) ;	

		memset(f,0,sizeof(*f));

		f->type = V4L2_TUNER_ANALOG_TV;

		f->frequency = btv->freq;

		// ST_TEST_1126
		// bttv_call_i2c_clients(btv,VIDIOC_G_FREQUENCY,f);


		return 0;
	}
	case VIDIOC_S_FREQUENCY:
	{
		struct v4l2_frequency *f = arg;

		//printk( "(tw_common_ioctls)----->VIDIOC_S_FREQUENCY\n" ) ;	

		if (unlikely(f->tuner != 0))
			return -EINVAL;
		if (unlikely (f->type != V4L2_TUNER_ANALOG_TV))
			return -EINVAL;
		mutex_lock(&btv->lock);
		
		btv->freq = f->frequency;
		
		bttv_call_i2c_clients(btv,VIDIOC_S_FREQUENCY,f);
		
		if (btv->has_matchbox && btv->radio_user)
			tea5757_set_freq(btv,btv->freq);
		
		mutex_unlock(&btv->lock);
		
		return 0;
	}
	case VIDIOC_LOG_STATUS:
	{
		DBG(KERN_INFO "bttv%d: =================  START STATUS CARD #%d  =================\n", btv->c.nr, btv->c.nr);
		bttv_call_i2c_clients(btv, VIDIOC_LOG_STATUS, NULL);
		DBG(KERN_INFO "bttv%d: ==================  END STATUS CARD #%d  ==================\n", btv->c.nr, btv->c.nr);
		return 0;
	}

	default:

		DBG( "**UNKNOW**\n" ) ;	
		
		return -ENOIOCTLCMD;

	}

	DBG( "<-----tw_common_ioctls \n" ) ;
	
	return 0;
	
}


// ST_ADD_0115
#if 0
static int
limit_scaled_size       (struct bttv_fh *               fh,
			 __s32 *                        width,
			 __s32 *                        height,
			 enum v4l2_field                field,
			 unsigned int			width_mask,
			 unsigned int			width_bias,
			 int                            adjust_size,
			 int                            adjust_crop)
{
	struct bttv *btv = fh->btv;
	const struct v4l2_rect *b;
	struct bttv_crop *c;
	__s32 min_width;
	__s32 min_height;
	__s32 max_width;
	__s32 max_height;
	int rc;

	/*
	BUG_ON((int) width_mask >= 0 ||
	       width_bias >= (unsigned int) -width_mask);
	*/

	/* Make sure tvnorm, vbi_end and the current cropping parameters
	   remain consistent until we're done. */
	mutex_lock(&btv->lock);

	b = &bttv_tvnorms[btv->tvnorm].cropcap.bounds;

	/* Do crop - use current, don't - use default parameters. */
	c = &btv->crop[!!fh->do_crop];

	if (fh->do_crop
	    && adjust_size
	    && adjust_crop
	    && !locked_btres(btv, VIDEO_RESOURCES)) {
		min_width = 48;
		min_height = 32;

		/* We cannot scale up. When the scaled image is larger
		   than crop.rect we adjust the crop.rect as required
		   by the V4L2 spec, hence cropcap.bounds are our limit. */
		max_width = min(b->width, (__s32) MAX_HACTIVE);
		max_height = b->height;

		/* We cannot capture the same line as video and VBI data.
		   Note btv->vbi_end is really a minimum, see
		   bttv_vbi_try_fmt(). */
		if (btv->vbi_end > b->top) {
			max_height -= btv->vbi_end - b->top;
			rc = -EBUSY;
			if (min_height > max_height)
				goto fail;
		}
	} else {
		rc = -EBUSY;
		if (btv->vbi_end > c->rect.top)
			goto fail;

		min_width  = c->min_scaled_width;
		min_height = c->min_scaled_height;
		max_width  = c->max_scaled_width;
		max_height = c->max_scaled_height;

		adjust_crop = 0;
	}

	min_width = (min_width - width_mask - 1) & width_mask;
	max_width = max_width & width_mask;

	/* Max. scale factor is 16:1 for frames, 8:1 for fields. */
	min_height = min_height;
	/* Min. scale factor is 1:1. */
	max_height >>= !V4L2_FIELD_HAS_BOTH(field);

	if (adjust_size) {
		*width = clamp(*width, min_width, max_width);
		*height = clamp(*height, min_height, max_height);

		/* Round after clamping to avoid overflow. */
		*width = (*width + width_bias) & width_mask;

		if (adjust_crop) {
			bttv_crop_adjust(c, b, *width, *height, field);

			if (btv->vbi_end > c->rect.top) {
				/* Move the crop window out of the way. */
				c->rect.top = btv->vbi_end;
			}
		}
	} else {
		rc = -EINVAL;
		if (*width  < min_width ||
		    *height < min_height ||
		    *width  > max_width ||
		    *height > max_height ||
		    0 != (*width & ~width_mask))
			goto fail;
	}

	rc = 0; /* success */

 fail:
	mutex_unlock(&btv->lock);

	return rc;
}

#endif



static int 
verify_window
(
	const struct bttv_tvnorm *tvn,
	struct v4l2_window *win, 
	int fixup
)
{
	enum v4l2_field field;
	int maxw, maxh;


	DBG("----->verify_window( w_width: %d, w_height: %d ) \n", win->w.width ,win->w.height) ;

	if (win->w.width  < 48 || win->w.height < 32)
		return -EINVAL;
	if (win->clipcount > 2048)
		return -EINVAL;

	field = win->field;
	maxw  = tvn->swidth;
	maxh  = tvn->sheight;

	if (V4L2_FIELD_ANY == field) {
		field = (win->w.height > maxh/2)
			? V4L2_FIELD_INTERLACED
			: V4L2_FIELD_TOP;
	}
	
	switch (field) {
	case V4L2_FIELD_TOP:
	case V4L2_FIELD_BOTTOM:
		maxh = maxh / 2;
		break;
	case V4L2_FIELD_INTERLACED:
		break;

	// ST_0121
	case V4L2_FIELD_INTERLACED_TB:
		DBG("verify_window::V4L2_FIELD_ALTERNATE_TB( w_width: %d, w_height: %d ) \n", win->w.width ,win->w.height) ;
		break ;
	// 
	
	default:
		return -EINVAL;
	}

	if (!fixup && (win->w.width > maxw || win->w.height > maxh))
		return -EINVAL;

	if (win->w.width > maxw) win->w.width = maxw ;
	if (win->w.height > maxh) win->w.height = maxh ;
	win->field = field;
	return 0;
	
}



static int 
setup_window
(
	struct bttv_fh *fh, 
	struct bttv *btv,
	struct v4l2_window *win, 
	int fixup
)
{
	struct v4l2_clip *clips = NULL;
	int n,size,retval = 0;

	DBG_SWITCH("----->setup_window\n") ;

	if (NULL == fh->ovfmt)
		return -EINVAL;
	
	if (!(fh->ovfmt->flags & FORMAT_FLAGS_PACKED))
		return -EINVAL;
	
	retval = verify_window(&bttv_tvnorms[btv->tvnorm],win,fixup);
	
	if (0 != retval)
		return retval;

	/* copy clips  --  luckily v4l1 + v4l2 are binary
	   compatible here ...*/
	n = win->clipcount;
	
	size = sizeof(*clips)*(n+4);
	
	clips = kmalloc(size,GFP_KERNEL);
	
	if (NULL == clips)
		return -ENOMEM;
	
	if (n > 0) {
		if (copy_from_user(clips,win->clips,sizeof(struct v4l2_clip)*n)) {
			kfree(clips);
			return -EFAULT;
		}
	}
	/* clip against screen */
	if (NULL != btv->fbuf.base)
		n = btcx_screen_clips(btv->fbuf.fmt.width, btv->fbuf.fmt.height,
				      &win->w, clips, n);
	btcx_sort_clips(clips,n);

	/* 4-byte alignments */
	switch (fh->ovfmt->depth) {
	case 8:
	case 24:
		btcx_align(&win->w, clips, n, 3);
		break;
	case 16:
		btcx_align(&win->w, clips, n, 1);
		break;
	case 32:
		/* no alignment fixups needed */
		break;
	default:
		BUG();
	}

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
	mutex_lock(&fh->cap.vb_lock);
#else
	mutex_lock(&fh->cap.lock);
#endif

	kfree(fh->ov.clips);
	fh->ov.clips    = clips;
	fh->ov.nclips   = n;

	fh->ov.w        = win->w;
	fh->ov.field    = win->field;
	fh->ov.setup_ok = 1;
	btv->init.ov.w.width   = win->w.width;
	btv->init.ov.w.height  = win->w.height;
	btv->init.ov.field     = win->field;

	/* update overlay if needed */
	retval = 0;
	if (check_btres(fh, RESOURCE_OVERLAY)) {
		struct bttv_buffer *new;

// ST_0303_ADD
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		new = videobuf_sg_alloc(sizeof(*new));
#else
		new = videobuf_alloc(sizeof(*new));
#endif
		bttv_overlay_risc(btv, &fh->ov, fh->ovfmt, new);
		retval = bttv_switch_overlay(btv,fh,new);
	}

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)		
	mutex_unlock(&fh->cap.vb_lock);
#else
	mutex_unlock(&fh->cap.lock);
#endif
	return retval;
}

/* ----------------------------------------------------------------------- */

static struct videobuf_queue* bttv_queue(struct bttv_fh *fh)
{
	struct videobuf_queue* q = NULL;

	switch (fh->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		q = &fh->cap;
		break;
	case V4L2_BUF_TYPE_VBI_CAPTURE:
		q = &fh->vbi;
		break;
	default:
		BUG();
	}
	return q;
}




static int 
tw_resource
(
	struct bttv_fh *fh
)
{
	int res = 0;
	struct bttv *btv = fh->btv;

	switch (fh->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		DBG_SWITCH( "tw_source ---> RESOURCE_VIDEO...\n" ) ;			
		res = RESOURCE_VIDEO;
		break;
	case V4L2_BUF_TYPE_VBI_CAPTURE:
		DBG( "tw_source ---> VBI\n" ) ;		
		res = RESOURCE_VBI;
		break;
	default:
		BUG();
	}
	return res;
}



// ST_TEST_0318 for VBI support 
#if 0

static int tw_switch_type(struct bttv_fh *fh, enum v4l2_buf_type type)
{
	struct videobuf_queue *q = bttv_queue(fh);
	int res = tw_resource(fh);

	if (check_btres(fh,res)){
		printk( "BUSY.....\n" ) ;
		return -EBUSY;
	}

	if (videobuf_queue_is_busy(q)){
		printk( "BUSY.....\n" ) ;
		return -EBUSY;
	}
	
	fh->type = type;

	return 0;
}

#else 

static int 
tw_switch_type
(
	struct bttv_fh *fh, 
	enum v4l2_buf_type type
)
{
	struct videobuf_queue *q = bttv_queue(fh);
	int res = tw_resource(fh);

	if (check_btres(fh,res))
		return -EBUSY;
	
	if (videobuf_queue_is_busy(q))
		return -EBUSY;
	
	fh->type = type;
	
	return 0;

}

#endif



static void
pix_format_set_size
(
	struct v4l2_pix_format *f,
	const struct bttv_format *fmt,
	unsigned int width,
	unsigned int height
)
{
	f->width = width;
	f->height = height;

	if (fmt->flags & FORMAT_FLAGS_PLANAR) {

		DBG("--->pix_format_set_size: Flags is FORMAT_FLAGS_PLANAR... \n" ) ;
		f->bytesperline = width; /* Y plane */
		f->sizeimage = (width * height * fmt->depth) >> 3;

	} else {
		DBG("--->pix_format_set_size: Flags is NOT FORMAT_FLAGS_PLANAR... \n" ) ;
		f->bytesperline = (width * fmt->depth) >> 3;
		f->sizeimage = height * f->bytesperline ;
	}


	// printk( "pix_format_set_size: flag: %x, width: %d, height: %d, bpl: %d, si: %d \n", fmt->flags, width, height, f->bytesperline, f->sizeimage  ) ;
	

}



static int 
bttv_g_fmt
(
	struct bttv_fh *fh, 
	struct v4l2_format *f
)
{
	
	DBG( "----->bttv_g_fmt \n " ) ;

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:

		DBG( "V4L2_BUF_TYPE_VIDEO_CAPTURE \n" ) ;

		memset(&f->fmt.pix,0,sizeof(struct v4l2_pix_format));

		pix_format_set_size (	&f->fmt.pix, 
							fh->fmt,
				     			fh->width, 
				     			fh->height);
		
		f->fmt.pix.field        = fh->cap.field;

		f->fmt.pix.pixelformat  = fh->fmt->fourcc;

		return 0;

	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		memset(&f->fmt.win,0,sizeof(struct v4l2_window));
		f->fmt.win.w     = fh->ov.w;
		f->fmt.win.field = fh->ov.field;
		return 0;
	case V4L2_BUF_TYPE_VBI_CAPTURE:

		DBG( "V4L2_BUF_TYPE_VBI_CAPTURE( bttv_vbi_get_fmt ) \n" ) ;
		
		bttv_vbi_get_fmt(fh,f);

		return 0;
	default:
		return -EINVAL;
	}
}




#if 1
static int 
bttv_try_fmt
(
	struct bttv_fh *fh, 
	struct bttv *btv,
	struct v4l2_format *f
)
{

	// ST_11172008, Force RISC program for single field ...
	int	VideoCaptureControl = 0; 
	//


	DBG( "----->bttv_try_fmt \n" ) ;	

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	{
		const struct bttv_format *fmt;
		enum v4l2_field field;
		unsigned int maxw,maxh;

		fmt = format_by_fourcc(f->fmt.pix.pixelformat);

		if (NULL == fmt)
			return -EINVAL;

		DBG( "Get format: %x \n", fmt ) ;

		/* fixup format */
		maxw  = bttv_tvnorms[btv->tvnorm].swidth;
		maxh  = bttv_tvnorms[btv->tvnorm].sheight;
		field = f->fmt.pix.field;

		DBG_SWITCH("--->bttv_try_fmt: Field:%d, swidth: %d, sheight: %d, pix_height: %d \n", 
				field, 
				maxw, 
				maxh, 
				f->fmt.pix.height ) ;

		// ST_MOD_0116
#if 0
		if (V4L2_FIELD_ANY == field)
			field = (f->fmt.pix.height > maxh/2)	? V4L2_FIELD_INTERLACED : V4L2_FIELD_BOTTOM;

#else

		if( bttv_tvnorms[btv->tvnorm].v4l2_id == V4L2_STD_PAL ) {
			if (V4L2_FIELD_ANY == field)
				field = V4L2_FIELD_INTERLACED ;
		} else {
			if (V4L2_FIELD_ANY == field)
				field = (f->fmt.pix.height > maxh/2)	? V4L2_FIELD_INTERLACED : V4L2_FIELD_BOTTOM;
		}
#endif

		if (V4L2_FIELD_ANY == field)
			field = V4L2_FIELD_INTERLACED ;
		
		if (V4L2_FIELD_SEQ_BT == field)
			field = V4L2_FIELD_SEQ_TB;
		
		switch (field) {
		case V4L2_FIELD_TOP:
			
			// ST_11172008, Force the RISC program for single field ...
			VideoCaptureControl |= ( TW6801_CAP_CTL_ODD | TW6801_CAP_CTL_EVEN ) ;
			
			DBG_SWITCH( "===== V4L2_FIELD_TOP ===== \n" ) ;	
			
			//
			maxh = maxh/2;
			break ;
			
		case V4L2_FIELD_BOTTOM:

			// ST_11172008, Force the RISC program for single field ...
			VideoCaptureControl |= TW6801_CAP_CTL_EVEN ;
			
			DBG_SWITCH( "===== V4L2_FIELD_BOTTOM ===== \n" ) ;	

			//
			maxh = maxh/2;
			
			break ;
			
		case V4L2_FIELD_ALTERNATE:

			// ST_11172008, Force the RISC program for single field ...
			VideoCaptureControl |= ( TW6801_CAP_CTL_ODD | TW6801_CAP_CTL_EVEN ) ;
			
			DBG_SWITCH( "===== V4L2_FIELD_ALTERNATE ===== \n" ) ;	


			//
			maxh = maxh/2;
			
			break;

		// ST_0121
		case V4L2_FIELD_INTERLACED_TB:
		case V4L2_FIELD_INTERLACED_BT:	
 			field = V4L2_FIELD_INTERLACED ;
			VideoCaptureControl |= ( TW6801_CAP_CTL_ODD | TW6801_CAP_CTL_EVEN ) ;
			DBG_SWITCH( "===== V4L2_FIELD_ALTERNATE_TB / BT===== \n" ) ;	
			break;			
		// 
			
		case V4L2_FIELD_INTERLACED:

			// ST_11172008, Force the RISC program for single field ...
			VideoCaptureControl |= ( TW6801_CAP_CTL_ODD | TW6801_CAP_CTL_EVEN ) ;
			DBG_SWITCH( "===== V4L2_FIELD_INTERLACED ===== \n" ) ;		
			//

			break;
			
		case V4L2_FIELD_SEQ_TB:
			if (fmt->flags & FORMAT_FLAGS_PLANAR)
				return -EINVAL;
			break;
		default:
			return -EINVAL;
			
		}

		// ST_11172008, force RISC program for single field ...
		// We should remove the setting in "tw_set_dma()" ...
		btaor( VideoCaptureControl, ~0x03, TW6801_CAP_CTL ) ;	
		//


		/* update data for the application */
		f->fmt.pix.field = field;
		
		if (f->fmt.pix.width  < 48)
			f->fmt.pix.width  = 48;

		if (f->fmt.pix.height < 32 )
			f->fmt.pix.height = 32;

		// ST 0115
#if 0
		if (f->fmt.pix.height < 289 /*244*/ ){
			printk( "Got wrong value for height: %d \n", f->fmt.pix.height ) ;
			return(-EINVAL) ;
		}
#endif

		if (f->fmt.pix.width  > maxw)
			f->fmt.pix.width = maxw;

		if (f->fmt.pix.height > maxh)
			f->fmt.pix.height = maxh;

		pix_format_set_size (	&f->fmt.pix, 
							fmt,
				     			f->fmt.pix.width & ~3,
				     			f->fmt.pix.height ) ;


		return 0;
		
	}
	
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		return verify_window(&bttv_tvnorms[btv->tvnorm],
				     &f->fmt.win, 1);
	case V4L2_BUF_TYPE_VBI_CAPTURE:

		DBG( "V4L2_BUF_TYPE_VBI_CAPTURE(bttv_vbi_try_fmt() \n" ) ;
		
		bttv_vbi_try_fmt(fh,f);

		return 0;
	default:
		return -EINVAL;
	}
}

#endif



static int 
bttv_s_fmt
(
	struct bttv_fh *fh, 
	struct bttv *btv,
      struct v4l2_format *f
)
{
	int retval;

	DBG_SWITCH( "----->bttv_s_fmt \n" ) ;
	
	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	{
		const struct bttv_format *fmt;

		DBG_SWITCH( "V4L2_BUF_TYPE_VIDEO_CAPTURE: %x \n", f->fmt.pix.pixelformat ) ;

		retval = tw_switch_type(fh,f->type);
		
		if (0 != retval)
			return retval;

		retval = bttv_try_fmt(fh,btv,f);

		if (0 != retval)
			return retval;

		fmt = format_by_fourcc(f->fmt.pix.pixelformat);

		/* update our state informations */
// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif

		fh->fmt              = fmt;
		fh->cap.field        = f->fmt.pix.field;
		fh->cap.last         = V4L2_FIELD_NONE;
		fh->width            = f->fmt.pix.width;
		fh->height           = f->fmt.pix.height;
		btv->init.fmt        = fmt;
		btv->init.width      = f->fmt.pix.width;
		btv->init.height     = f->fmt.pix.height;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif
		return 0;
	}
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		if (no_overlay > 0) {
			DBG("V4L2_BUF_TYPE_VIDEO_OVERLAY: no_overlay\n");
			return -EINVAL;
		}

		return setup_window(fh, btv, &f->fmt.win, 1);

	case V4L2_BUF_TYPE_VBI_CAPTURE:

		DBG( "V4L2_BUF_TYPE_VBI_CAPTURE(tw_switch_type) \n " ) ;

		retval = tw_switch_type(fh,f->type);
		
		if (0 != retval) {
			DBG( "V4L2_BUF_TYPE_VBI_CAPTURE got error -1 ...\n" ) ;
			return retval;
		}

		if (locked_btres(fh->btv, RESOURCE_VBI)) {
			DBG( "V4L2_BUF_TYPE_VBI_CAPTURE got error -2 ...\n" ) ;
			return -EBUSY;
		}

		bttv_vbi_try_fmt(fh,f);
		bttv_vbi_setlines(fh,btv,f->fmt.vbi.count[0]);
		bttv_vbi_get_fmt(fh,f);
		return 0;
		
	default:
		DBG( "V4L2_BUF_TYPE_VBI_CAPTURE got error -3 ...\n" ) ;	
		return -EINVAL;
	}


}




static int tw_do_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, void *arg)
{
	struct bttv_fh *fh  = file->private_data;
	struct bttv    *btv = fh->btv;
	unsigned long flags;
	int retval = 0;


	DBG( "----->tw_do_ioctl ( cmd: %x ) \n" , cmd) ;

	if (bttv_debug > 1)
		v4l_print_ioctl(btv->c.name, cmd);

	if (btv->errors) {
		DBG( "tw_do_ioctl--->error \n" ) ;
		bttv_reinit_tw680x(btv);
	}

	switch (cmd) {
	case VIDIOCSFREQ:
	case VIDIOCSTUNER:
	case VIDIOCSCHAN:
	case VIDIOC_S_CTRL:
	case VIDIOC_S_STD:
	case VIDIOC_S_INPUT:
	case VIDIOC_S_TUNER:
	case VIDIOC_S_FREQUENCY:
		retval = v4l2_prio_check(&btv->prio,&fh->prio);
		if (0 != retval)
			return retval;
	};

	switch (cmd) {

	/* ***  v4l1  *** ************************************************ */
	case VIDIOCGCAP:
	{
		struct video_capability *cap = arg;

		DBG_SWITCH( "**VIDIOCGCAP**\n" ) ;

		memset(cap,0,sizeof(*cap));
		strcpy(cap->name,btv->video_dev->name);
		if (V4L2_BUF_TYPE_VBI_CAPTURE == fh->type) {
			/* vbi */
			cap->type = VID_TYPE_TUNER|VID_TYPE_TELETEXT;
		} else {
			/* others */
			cap->type = VID_TYPE_CAPTURE|
				VID_TYPE_TUNER|
				VID_TYPE_CLIPPING|
				VID_TYPE_SCALES;
			if (no_overlay <= 0)
				cap->type |= VID_TYPE_OVERLAY;

			cap->maxwidth  = bttv_tvnorms[btv->tvnorm].swidth;
			cap->maxheight = bttv_tvnorms[btv->tvnorm].sheight;
			cap->minwidth  = 48;
			cap->minheight = 32;
		}
		cap->channels  = bttv_tvcards[btv->c.type].video_inputs;
		cap->audios    = bttv_tvcards[btv->c.type].audio_inputs;
		return 0;
	}

	case VIDIOCGPICT:
	{
		struct video_picture *pic = arg;

		DBG( "**VIDIOCGPICT**\n" ) ;

		memset(pic,0,sizeof(*pic));
		pic->brightness = btv->bright;
		pic->contrast   = btv->contrast;
		pic->hue        = btv->hue;
		pic->colour     = btv->saturation;
		if (fh->fmt) {
			pic->depth   = fh->fmt->depth;
			pic->palette = fh->fmt->palette;
		}
		return 0;
	}
	case VIDIOCSPICT:
	{
		struct video_picture *pic = arg;
		const struct bttv_format *fmt;

		DBG( "**VIDIOCSPICT**\n" ) ;
		
		fmt = format_by_palette(pic->palette);
		if (NULL == fmt)
			return -EINVAL;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif

		if (fmt->depth != pic->depth) {
			retval = -EINVAL;
			goto fh_unlock_and_return;
		}
		if (fmt->flags & FORMAT_FLAGS_RAW) {
			/* VIDIOCMCAPTURE uses gbufsize, not RAW_BPL *
			   RAW_LINES * 2. F1 is stored at offset 0, F2
			   at buffer size / 2. */
			fh->width = RAW_BPL;
			fh->height = gbufsize / RAW_BPL;
			btv->init.width  = RAW_BPL;
			btv->init.height = gbufsize / RAW_BPL;
		}
		fh->ovfmt   = fmt;
		fh->fmt     = fmt;
		btv->init.ovfmt   = fmt;
		btv->init.fmt     = fmt;
		if (bigendian) {
			/* dirty hack time:  swap bytes for overlay if the
			   display adaptor is big endian (insmod option) */
			if (fmt->palette == VIDEO_PALETTE_RGB555 ||
			    fmt->palette == VIDEO_PALETTE_RGB565 ||
			    fmt->palette == VIDEO_PALETTE_RGB32) {
				fh->ovfmt = fmt+1;
			}
		}

		/*
		bt848_bright(btv,pic->brightness);
		bt848_contrast(btv,pic->contrast);
		bt848_hue(btv,pic->hue);
		bt848_sat(btv,pic->colour);
		*/	
		
		TW6801_bright(btv,pic->brightness);
		TW6801_contrast(btv,pic->contrast);
		TW6801_hue(btv,pic->hue);
		TW6801_sat(btv,pic->colour);

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif
		
		return 0;
	}

	case VIDIOCGWIN:
	{
		struct video_window *win = arg;

		DBG( "**VIDIOCGWIN**\n" ) ;

		memset(win,0,sizeof(*win));
		win->x      = fh->ov.w.left;
		win->y      = fh->ov.w.top;
		win->width  = fh->ov.w.width;
		win->height = fh->ov.w.height;
		return 0;
	}
	case VIDIOCSWIN:
	{
		struct video_window *win = arg;
		struct v4l2_window w2;

		DBG( "**VIDIOCSWIN**\n" ) ;

		if (no_overlay > 0) {
			DBG ("VIDIOCSWIN: no_overlay\n");
			return -EINVAL;
		}

		w2.field = V4L2_FIELD_ANY;
		w2.w.left    = win->x;
		w2.w.top     = win->y;
		w2.w.width   = win->width;
		w2.w.height  = win->height;
		w2.clipcount = win->clipcount;
		w2.clips     = (struct v4l2_clip __user *)win->clips;
		retval = setup_window(fh, btv, &w2, 0);
		if (0 == retval) {
			/* on v4l1 this ioctl affects the read() size too */
			fh->width  = fh->ov.w.width;
			fh->height = fh->ov.w.height;
			btv->init.width  = fh->ov.w.width;
			btv->init.height = fh->ov.w.height;
		}
		return retval;
	}

	case VIDIOCGFBUF:
	{
		struct video_buffer *fbuf = arg;

		DBG( "**VIDIOCGFBUF**\n" ) ;

		fbuf->base          = btv->fbuf.base;
		fbuf->width         = btv->fbuf.fmt.width;
		fbuf->height        = btv->fbuf.fmt.height;
		fbuf->bytesperline  = btv->fbuf.fmt.bytesperline;
		if (fh->ovfmt)
			fbuf->depth = fh->ovfmt->depth;
		return 0;
	}
	case VIDIOCSFBUF:
	{
		struct video_buffer *fbuf = arg;
		const struct bttv_format *fmt;
		unsigned long end;

		DBG( "**VIDIOCSFBUF**\n" ) ;		

		if(!capable(CAP_SYS_ADMIN) &&
		   !capable(CAP_SYS_RAWIO))
			return -EPERM;
		end = (unsigned long)fbuf->base +
			fbuf->height * fbuf->bytesperline;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif

		retval = -EINVAL;

		switch (fbuf->depth) {
		case 8:
			fmt = format_by_palette(VIDEO_PALETTE_HI240);
			break;
		case 16:
			fmt = format_by_palette(VIDEO_PALETTE_RGB565);
			break;
		case 24:
			fmt = format_by_palette(VIDEO_PALETTE_RGB24);
			break;
		case 32:
			fmt = format_by_palette(VIDEO_PALETTE_RGB32);
			break;
		case 15:
			fbuf->depth = 16;
			fmt = format_by_palette(VIDEO_PALETTE_RGB555);
			break;
		default:
			fmt = NULL;
			break;
		}
		if (NULL == fmt)
			goto fh_unlock_and_return;

		fh->ovfmt = fmt;
		fh->fmt   = fmt;
		btv->init.ovfmt = fmt;
		btv->init.fmt   = fmt;
		btv->fbuf.base             = fbuf->base;
		btv->fbuf.fmt.width        = fbuf->width;
		btv->fbuf.fmt.height       = fbuf->height;
		if (fbuf->bytesperline)
			btv->fbuf.fmt.bytesperline = fbuf->bytesperline;
		else
			btv->fbuf.fmt.bytesperline = btv->fbuf.fmt.width*fbuf->depth/8;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif

		return 0;
	}

	case VIDIOCCAPTURE:
	case VIDIOC_OVERLAY:
	{
		struct bttv_buffer *new;
		int *on = arg;

		DBG( "**VIDIOCCAPTURE**\n" ) ;

		if (*on) {
			/* verify args */
			if (NULL == btv->fbuf.base)
				return -EINVAL;
			if (!fh->ov.setup_ok) {
				dprintk("bttv%d: overlay: !setup_ok\n",btv->c.nr);
				return -EINVAL;
			}
		}

		if (!check_alloc_btres(btv,fh,RESOURCE_OVERLAY))
			return -EBUSY;

		// ST_0122_ADD
		if (!check_alloc_btres(btv,fh,RESOURCE_VIDEO))
			return -EBUSY;


// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif		

		if (*on) {
			fh->ov.tvnorm = btv->tvnorm;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)		
			new = videobuf_sg_alloc(sizeof(*new));
#else
			new = videobuf_alloc(sizeof(*new));
#endif
			bttv_overlay_risc(btv, &fh->ov, fh->ovfmt, new);
		} else {
			new = NULL;
		}

		/* switch over */
		retval = bttv_switch_overlay(btv,fh,new);

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)		
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif

		return retval;
		
	}

	case VIDIOCGMBUF:
	{
		struct video_mbuf *mbuf = arg;
		unsigned int i;

		DBG( "**VIDIOCGMBUF**\n" ) ;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else		
		mutex_lock(&fh->cap.lock);
#endif
		
		retval = videobuf_mmap_setup(&fh->cap,gbuffers,gbufsize,
					     V4L2_MEMORY_MMAP);
		if (retval < 0)
			goto fh_unlock_and_return;
		memset(mbuf,0,sizeof(*mbuf));
		mbuf->frames = gbuffers;
		mbuf->size   = gbuffers * gbufsize;
		for (i = 0; i < gbuffers; i++)
			mbuf->offsets[i] = i * gbufsize;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif

		return 0;
	}
	case VIDIOCMCAPTURE:
	{
		struct video_mmap *vm = arg;
		struct bttv_buffer *buf;
		enum v4l2_field field;

		DBG_SWITCH("****VIDIOCMCAPTURE****\n" ) ;

		if (vm->frame >= VIDEO_MAX_FRAME)
			return -EINVAL;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif		

		retval = -EINVAL;
		
		buf = (struct bttv_buffer *)fh->cap.bufs[vm->frame];
		
		if (NULL == buf)
			goto fh_unlock_and_return;
		
		if (0 == buf->vb.baddr)
			goto fh_unlock_and_return;

		if (buf->vb.state == STATE_QUEUED ||
		    buf->vb.state == STATE_ACTIVE)
			goto fh_unlock_and_return;

		// ST_0121
		// field = (vm->height > bttv_tvnorms[btv->tvnorm].sheight/2)
		field = (vm->height >= bttv_tvnorms[btv->tvnorm].sheight/2)
			? V4L2_FIELD_INTERLACED
			: V4L2_FIELD_BOTTOM;

		DBG_SWITCH( "height: %d, sheight: %d, Field: %d, tvnorm: %d \n", 
				vm->height,
				bttv_tvnorms[btv->tvnorm].sheight/2,
				field, 
				btv->tvnorm ) ;
		
		retval = tw_prepare_buffer(	&fh->cap,
								btv,
								buf,
					     			format_by_palette(vm->format),
					     			vm->width,
					     			vm->height,
					     			field ) ;
		
		if (0 != retval)
			goto fh_unlock_and_return;

		spin_lock_irqsave( &btv->s_lock,flags ) ;

		buffer_queue( &fh->cap, &buf->vb ) ;

		spin_unlock_irqrestore(&btv->s_lock,flags);

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif

		return 0;
		
	}
	case VIDIOCSYNC:
	{
		int *frame = arg;
		struct bttv_buffer *buf;

		DBG( "****VIDIOCSYNC****\n" ) ;

		if (*frame >= VIDEO_MAX_FRAME) return -EINVAL ;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif

		retval = -EINVAL;
		buf = (struct bttv_buffer *)fh->cap.bufs[*frame];
		if (NULL == buf)
			goto fh_unlock_and_return;
		retval = videobuf_waiton(&buf->vb,0,1);
		if (0 != retval)
			goto fh_unlock_and_return;
		switch (buf->vb.state) {
		case STATE_ERROR:
			retval = -EIO;
			/* fall through */
		case STATE_DONE:
		{
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
			struct videobuf_dmabuf *dma=videobuf_to_dma(&buf->vb);
			videobuf_dma_sync(&fh->cap,dma);

#else

			videobuf_dma_sync(&fh->cap,&buf->vb.dma);
#endif

			tw_dma_free(&fh->cap,btv,buf);

			break;
		}
		default:
			retval = -EINVAL;
			break;
		}

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif

		return retval;
		
	}

	case VIDIOCGVBIFMT:
	{

// ST_TEST_0318 for VBI support 
#if 0		
		struct vbi_format *fmt = (void *) arg;
		struct v4l2_format fmt2;

		printk( "===== VIDIOCGVBIFMT ===== \n" ) ;

		if (fh->type != V4L2_BUF_TYPE_VBI_CAPTURE) {

			printk( "----->tw_switch_type to VBI_CAPTURE \n" ) ;

			retval = tw_switch_type(fh,V4L2_BUF_TYPE_VBI_CAPTURE);

			if (0 != retval) {
				printk( "<-----tw_switch_type return error \n" ) ;
				return retval;
			}

		}

		bttv_vbi_get_fmt(fh, &fmt2);

		memset(fmt,0,sizeof(*fmt));
		fmt->sampling_rate    	= fmt2.fmt.vbi.sampling_rate;
		fmt->samples_per_line 	= fmt2.fmt.vbi.samples_per_line;
		fmt->sample_format    	= VIDEO_PALETTE_RAW;
		fmt->start[0]         		= fmt2.fmt.vbi.start[0];
		fmt->count[0]         		= fmt2.fmt.vbi.count[0];
		fmt->start[1]         		= fmt2.fmt.vbi.start[1];
		fmt->count[1]         		= fmt2.fmt.vbi.count[1];

		if (fmt2.fmt.vbi.flags & V4L2_VBI_UNSYNC)
			fmt->flags   |= VBI_UNSYNC;
		
		if (fmt2.fmt.vbi.flags & V4L2_VBI_INTERLACED)
			fmt->flags   |= VBI_INTERLACED;

		printk( "<-----tw_switch_type -2 \n" ) ;
		
		return 0;
#else
		struct vbi_format *fmt = (void *) arg;
		struct v4l2_format fmt2;

		if (fh->type != V4L2_BUF_TYPE_VBI_CAPTURE) {
			retval = tw_switch_type(fh,V4L2_BUF_TYPE_VBI_CAPTURE);
			if (0 != retval)
				return retval;
		}
		bttv_vbi_get_fmt(fh, &fmt2);

		memset(fmt,0,sizeof(*fmt));
		fmt->sampling_rate    = fmt2.fmt.vbi.sampling_rate;
		fmt->samples_per_line = fmt2.fmt.vbi.samples_per_line;
		fmt->sample_format    = VIDEO_PALETTE_RAW;
		fmt->start[0]         = fmt2.fmt.vbi.start[0];
		fmt->count[0]         = fmt2.fmt.vbi.count[0];
		fmt->start[1]         = fmt2.fmt.vbi.start[1];
		fmt->count[1]         = fmt2.fmt.vbi.count[1];
		if (fmt2.fmt.vbi.flags & VBI_UNSYNC)
			fmt->flags   |= V4L2_VBI_UNSYNC;
		if (fmt2.fmt.vbi.flags & VBI_INTERLACED)
			fmt->flags   |= V4L2_VBI_INTERLACED;
		return 0;


#endif


		
	}
	case VIDIOCSVBIFMT:
	{

#if 0
		struct vbi_format *fmt = (void *) arg;
		struct v4l2_format fmt2;

		printk( "**VIDIOCSVBIFMT**\n" ) ;

		retval = tw_switch_type(fh,V4L2_BUF_TYPE_VBI_CAPTURE);

		if (0 != retval){
			printk( "VIDIOCSVBIFMT-0 ....\n" ) ;
			return retval;
		}

		bttv_vbi_get_fmt(fh, &fmt2);

#if 1	
		if (fmt->sampling_rate    		!= fmt2.fmt.vbi.sampling_rate     ||
		    fmt->samples_per_line 	!= fmt2.fmt.vbi.samples_per_line  ||
		    fmt->sample_format    		!= VIDEO_PALETTE_RAW              ||
		    // ST_DEL_0316 for VBI Support 	
		    /*fmt->start[0]         			!= fmt2.fmt.vbi.start[0]          ||
		    fmt->start[1]         			!= fmt2.fmt.vbi.start[1]          ||*/
	
		    fmt->count[0]         		!= fmt->count[1]                  ||
		    fmt->count[0]         		<  1                              ||
		    fmt->count[0]         		>  32  /* VBI_MAXLINES */) {

			printk( "===== VIDIOCSVBIFMT -EINVAL=====.\n" ) ;
				
			printk( "sample_rate: %d == %d\n",  fmt->sampling_rate , fmt2.fmt.vbi.sampling_rate ) ;
			printk( "sample_per_line: %d == %d\n",  fmt->samples_per_line , fmt2.fmt.vbi.samples_per_line ) ;
			printk( "sample_format: %d == %d\n",  fmt->sample_format , VIDEO_PALETTE_RAW ) ;
			printk( "sample_start_0: %d == %d\n",  fmt->start[0] , fmt2.fmt.vbi.start[0] ) ;				
			printk( "sample_start_1: %d == %d\n",  fmt->start[1] , fmt2.fmt.vbi.start[1] ) ;
			printk( "sample_count0: %d\n",  fmt->count[0]  ) ;
			printk( "sample_count1: %d\n",  fmt->count[1] ) ;
	
			return -EINVAL;
		}

#else 
			printk( "sample_rate: %d == %d\n",  fmt->sampling_rate , fmt2.fmt.vbi.sampling_rate ) ;
			printk( "sample_per_line: %d == %d\n",  fmt->samples_per_line , fmt2.fmt.vbi.samples_per_line ) ;
			printk( "sample_format: %d == %d\n",  fmt->sample_format , VIDEO_PALETTE_RAW ) ;
			printk( "sample_start_0: %d == %d\n",  fmt->start[0] , fmt2.fmt.vbi.start[0] ) ;				
			printk( "sample_start_1: %d == %d\n",  fmt->start[1] , fmt2.fmt.vbi.start[1] ) ;
			printk( "sample_count0: %d\n",  fmt->count[0]  ) ;
			printk( "sample_count1: %d\n",  fmt->count[1] ) ;


#endif




		bttv_vbi_setlines(fh,btv,fmt->count[0]) ;
		
		return 0 ;
#else

		struct vbi_format *fmt = (void *) arg;
		struct v4l2_format fmt2;

		retval = tw_switch_type(fh,V4L2_BUF_TYPE_VBI_CAPTURE);
		if (0 != retval)
			return retval;
		bttv_vbi_get_fmt(fh, &fmt2);

		if (fmt->sampling_rate    != fmt2.fmt.vbi.sampling_rate     ||
		    fmt->samples_per_line != fmt2.fmt.vbi.samples_per_line  ||
		    fmt->sample_format    != VIDEO_PALETTE_RAW              ||
		    fmt->start[0]         != fmt2.fmt.vbi.start[0]          ||
		    fmt->start[1]         != fmt2.fmt.vbi.start[1]          ||
		    fmt->count[0]         != fmt->count[1]                  ||
		    fmt->count[0]         <  1                              ||
		    fmt->count[0]         >  32 /* VBI_MAXLINES */)
			return -EINVAL;

		bttv_vbi_setlines(fh,btv,fmt->count[0]);
		return 0;


#endif


	}

	case BTTV_VERSION:
	case VIDIOCGFREQ:
	case VIDIOCSFREQ:
	case VIDIOCGTUNER:
	case VIDIOCSTUNER:
	case VIDIOCGCHAN:
	case VIDIOCSCHAN:
	case VIDIOCGAUDIO:
	case VIDIOCSAUDIO:
		return tw_common_ioctls(btv,cmd,arg);

	/* ***  v4l2  *** ************************************************ */
	case VIDIOC_QUERYCAP:
	{
		struct v4l2_capability *cap = arg;

		DBG( "\n===== VIDIOC_QUERYCAP =====\n" ) ;

		if (0 == v4l2) {
			DBG( "\n There is no V4L2 available ===> EINVAL \n" ) ;
			DBG( "\n\n" ) ;
			return -EINVAL;
		}

		memset(cap, 0, sizeof (*cap));
		strlcpy(cap->driver, "bttv", sizeof (cap->driver));
		strlcpy(cap->card, btv->video_dev->name, sizeof (cap->card));
		snprintf(cap->bus_info, sizeof (cap->bus_info),
			 "PCI:%s", pci_name(btv->c.pci));

		DBG( "Info: driver: %s, card: %s, bus_info: %s \n", cap->driver, cap->card, cap->bus_info ) ; 


		cap->version = TW6801_VERSION_CODE;

			cap->capabilities =	V4L2_CAP_VIDEO_CAPTURE |
							V4L2_CAP_READWRITE |
							V4L2_CAP_STREAMING ;

		if (no_overlay <= 0)
			cap->capabilities |= V4L2_CAP_VIDEO_OVERLAY;

		if (bttv_tvcards[btv->c.type].tuner != UNSET &&
		    bttv_tvcards[btv->c.type].tuner != TUNER_ABSENT)
			cap->capabilities |= V4L2_CAP_TUNER;

		DBG( "<-----return = 0 ....\n" ) ;
 		
		return 0;
	}

	case VIDIOC_ENUM_FMT:
	{
		struct v4l2_fmtdesc *f = arg;
		enum v4l2_buf_type type;
		unsigned int i;
		int index;

		type  = f->type;
		
		DBG( "\n ===== VIDIOC_ENUM_FMT (type: %x, index: %d)=====\n", type, f->index ) ;

		if (V4L2_BUF_TYPE_VBI_CAPTURE == type) {
			DBG( "----->VBI\n" ) ;		
	
			/* vbi */
			index = f->index;

			if (0 != index) {
				DBG( "Error ...\n " ) ;	
				return -EINVAL;
			}

			memset(f,0,sizeof(*f));
			f->index       = index;
			f->type        = type;
			f->pixelformat = V4L2_PIX_FMT_GREY;
			strcpy(f->description,"vbi data");
			return 0;
		}

		/* video capture + overlay */
		DBG( "Video Capture + overlay ...\n" ) ;
		index = -1;
		for (i = 0; i < BTTV_FORMATS; i++) {
			if (bttv_formats[i].fourcc != -1)
				index++;
			if ((unsigned int)index == f->index)
				break;
		}
		if (BTTV_FORMATS == i)
			return -EINVAL;

		switch (f->type) {
		case V4L2_BUF_TYPE_VIDEO_CAPTURE:
			break;
		case V4L2_BUF_TYPE_VIDEO_OVERLAY:
			if (!(bttv_formats[i].flags & FORMAT_FLAGS_PACKED))
				return -EINVAL;
			break;
		default:
			return -EINVAL;
		}
		memset(f,0,sizeof(*f));
		f->index       = index;
		f->type        = type;
		f->pixelformat = bttv_formats[i].fourcc;
		strlcpy(f->description,bttv_formats[i].name,sizeof(f->description));
		return 0;
	}

	case VIDIOC_TRY_FMT:
	{
		struct v4l2_format *f = arg;

		DBG( "**VIDIOC_TRY_FMT**\n" ) ;

		return bttv_try_fmt(fh,btv,f);
	}
	case VIDIOC_G_FMT:
	{
		struct v4l2_format *f = arg;

		DBG( "\n===== VIDIOC_G_FMT ====== \n" ) ;

		return bttv_g_fmt(fh,f);
	}
	case VIDIOC_S_FMT:
	{
		struct v4l2_format *f = arg;

		DBG( "**VIDIOC_S_FMT**\n" ) ;

		return bttv_s_fmt(fh,btv,f);
	}

	case VIDIOC_G_FBUF:
	{
		struct v4l2_framebuffer *fb = arg;

		DBG( "**VIDIOC_G_FBUF**\n" ) ;

		*fb = btv->fbuf;
		fb->capability = V4L2_FBUF_CAP_LIST_CLIPPING;
		if (fh->ovfmt)
			fb->fmt.pixelformat  = fh->ovfmt->fourcc;
		return 0;
	}
	case VIDIOC_S_FBUF:
	{
		struct v4l2_framebuffer *fb = arg;
		const struct bttv_format *fmt;

		DBG( "**VIDIOC_S_FBUF**\n" ) ;

		if(!capable(CAP_SYS_ADMIN) &&
		   !capable(CAP_SYS_RAWIO))
			return -EPERM;

		/* check args */
		fmt = format_by_fourcc(fb->fmt.pixelformat);
		if (NULL == fmt)
			return -EINVAL;
		if (0 == (fmt->flags & FORMAT_FLAGS_PACKED))
			return -EINVAL;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif

		retval = -EINVAL;
		if (fb->flags & V4L2_FBUF_FLAG_OVERLAY) {
			if (fb->fmt.width > bttv_tvnorms[btv->tvnorm].swidth)
				goto fh_unlock_and_return;
			if (fb->fmt.height > bttv_tvnorms[btv->tvnorm].sheight)
				goto fh_unlock_and_return;
		}

		/* ok, accept it */
		btv->fbuf.base       = fb->base;
		btv->fbuf.fmt.width  = fb->fmt.width;
		btv->fbuf.fmt.height = fb->fmt.height;
		if (0 != fb->fmt.bytesperline)
			btv->fbuf.fmt.bytesperline = fb->fmt.bytesperline;
		else
			btv->fbuf.fmt.bytesperline = btv->fbuf.fmt.width*fmt->depth/8;

		retval = 0;
		fh->ovfmt = fmt;
		btv->init.ovfmt = fmt;
		if (fb->flags & V4L2_FBUF_FLAG_OVERLAY) {
			fh->ov.w.left   = 0;
			fh->ov.w.top    = 0;
			fh->ov.w.width  = fb->fmt.width;
			fh->ov.w.height = fb->fmt.height;
			btv->init.ov.w.width  = fb->fmt.width;
			btv->init.ov.w.height = fb->fmt.height;
				kfree(fh->ov.clips);
			fh->ov.clips = NULL;
			fh->ov.nclips = 0;

			if (check_btres(fh, RESOURCE_OVERLAY)) {
				struct bttv_buffer *new;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
				new = videobuf_sg_alloc(sizeof(*new));
#else
				new = videobuf_alloc(sizeof(*new));
#endif
				bttv_overlay_risc(btv,&fh->ov,fh->ovfmt,new);
				retval = bttv_switch_overlay(btv,fh,new);
			}
		}

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_unlock(&fh->cap.vb_lock);
#else
		mutex_unlock(&fh->cap.lock);
#endif
		return retval;

	}

	case VIDIOC_REQBUFS:
	{
		DBG( "**VIDIOC_REQBUFS**\n" ) ;

		//ST_0122, "videobuf_reqbufs()" will call into "ops->buf_setup()"
		
		struct v4l2_requestbuffers *Request = (struct v4l2_requestbuffers *)(arg) ;

		DBG_SWITCH("Count: %d ...\n", Request->count ) ;
	
		return videobuf_reqbufs(bttv_queue(fh),arg);
	
		
	}
	case VIDIOC_QUERYBUF:
	{
		DBG( "**VIDIOC_QUERYBUF**\n" ) ;

		return videobuf_querybuf(bttv_queue(fh),arg);
	}
	case VIDIOC_QBUF:
	{
		DBG( "**VIDIOC_QBUF - %d**\n", fh->btv->c.nr ) ;
		// ST_0122_NOTE. This function will call into "buf_prepare"
		return videobuf_qbuf(bttv_queue(fh),arg);

	}
	case VIDIOC_DQBUF:
	{
		DBG( "**VIDIOC_DQBUF - %d**\n", fh->btv->c.nr ) ;

/* ST_MOD_0805 */
#if 0
		int	rval ;

__videobuf_dqbuf_again:
	
		rval = videobuf_dqbuf( bttv_queue(fh),arg, file->f_flags & O_NONBLOCK ) ;

		if ( rval ) goto __OUT ;

		return 0 ;

__OUT:

        	if (rval == -EIO) {
	
			DBG_FPS( "==> VIDIOC_DQBUF - %d ==== EIO ==== \n", fh->btv->c.nr ) ;

                	videobuf_qbuf( bttv_queue(fh), arg ) ;

                	if (!(file->f_flags & O_NONBLOCK)) goto __videobuf_dqbuf_again ;
                	/*
                	  * We don't have a videobuf_buffer now --- maybe next
                 	  * time...
                 	  */
                	rval = -EAGAIN;

			DBG_FPS( "==> VIDIOC_DQBUF - %d ==== EAGAIN ==== \n", fh->btv->c.nr ) ;

		}

		return rval ;


#else
		return videobuf_dqbuf( bttv_queue(fh),arg,
				      file->f_flags & O_NONBLOCK);

#endif

	}
	case VIDIOC_STREAMON:
	{

		int res ;

		DBG_FPS( "**VIDIOC_STREAMON**\n" ) ;

		init_irqreg(btv) ;

		mod_timer(&btv->timeout, jiffies+  msecs_to_jiffies(15000)  ) ;				

		tw_EnableDMA( btv ) ;

		res = tw_resource(fh);

		if (!check_alloc_btres(btv,fh,res))
			return -EBUSY;

		// ST_0419, V4L will call into buffer_queue() 
		return videobuf_streamon(bttv_queue(fh));
	}
	
	case VIDIOC_STREAMOFF:
	{

		int res ;

		DBG_FPS( "==== VIDIOC_STREAMOFF - %d ====\n", btv->c.nr ) ;

		btwrite( 0, TW6801_INT_MASK ) ;	
		btwrite(~0x0, TW6801_INTSTAT);

		tw_DisableDMA(btv) ;

		/* ST_090200, clear all buffer state to make sure all buffer was wake-up */
		tw_clear_buffer_state( btv ) ;	
		
			
		res = tw_resource(fh);

		retval = videobuf_streamoff(bttv_queue(fh));
		
		if (retval < 0) {
			DBG( "<----VIDIOC_STREAMOFF-%d, retval: %x !!!\n", btv->c.nr, retval ) ;
			return retval;
		}


		DBG( "videobuf_streamoff-%d, return OK !!!\n", btv->c.nr ) ;
		
		free_btres(btv,fh,res);

		DBG( "<----VIDIOC_STREAMOFF-%d, return OK !!!\n", btv->c.nr ) ;
	
		return 0;

	}

	case VIDIOC_QUERYCTRL:
	{
		struct v4l2_queryctrl *c = arg;
		int i;

		DBG( "**VIDIOC_QUERYCTRL**\n" ) ;
		
		if ((c->id <  V4L2_CID_BASE ||
		     c->id >= V4L2_CID_LASTP1) &&
		    (c->id <  V4L2_CID_PRIVATE_BASE ||
		     c->id >= V4L2_CID_PRIVATE_LASTP1))
			return -EINVAL;
		for (i = 0; i < BTTV_CTLS; i++)
			if (bttv_ctls[i].id == c->id)
				break;
		if (i == BTTV_CTLS) {
			*c = no_ctl;
			return 0;
		}
		*c = bttv_ctls[i];
		if (btv->audio_hook && i >= 4 && i <= 8) {
			struct video_audio va;
			memset(&va,0,sizeof(va));
			btv->audio_hook(btv,&va,0);
			switch (bttv_ctls[i].id) {
			case V4L2_CID_AUDIO_VOLUME:
				if (!(va.flags & VIDEO_AUDIO_VOLUME))
					*c = no_ctl;
				break;
			case V4L2_CID_AUDIO_BALANCE:
				if (!(va.flags & VIDEO_AUDIO_BALANCE))
					*c = no_ctl;
				break;
			case V4L2_CID_AUDIO_BASS:
				if (!(va.flags & VIDEO_AUDIO_BASS))
					*c = no_ctl;
				break;
			case V4L2_CID_AUDIO_TREBLE:
				if (!(va.flags & VIDEO_AUDIO_TREBLE))
					*c = no_ctl;
				break;
			}
		}
		return 0;
	}
	case VIDIOC_G_CTRL:
	{
		DBG( "**VIDIOC_G_CTRL**\n" ) ;
		
		return get_control(btv,arg);
	}
		
	case VIDIOC_S_CTRL:
	{
		DBG( "**VIDIOC_S_CTRL**\n" ) ;		
		return set_control(btv,arg);
	}
		
	case VIDIOC_G_PARM:
	{
		struct v4l2_streamparm *parm = arg;
		struct v4l2_standard s;
		
		DBG( "**VIDIOC_G_PARM**\n" ) ;	
		
		if (parm->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;
		memset(parm,0,sizeof(*parm));
		v4l2_video_std_construct(&s, bttv_tvnorms[btv->tvnorm].v4l2_id,
					 bttv_tvnorms[btv->tvnorm].name);
		parm->parm.capture.timeperframe = s.frameperiod;
		return 0;
	}

	case VIDIOC_G_PRIORITY:
	{
		enum v4l2_priority *p = arg;

		DBG( "**VIDIOC_G_PARM**\n" ) ;	
		
		*p = v4l2_prio_max(&btv->prio);
		return 0;
	}
	case VIDIOC_S_PRIORITY:
	{
		enum v4l2_priority *prio = arg;
		
		DBG( "**VIDIOC_S_PRIORITY**\n" ) ;	

		return v4l2_prio_change(&btv->prio, &fh->prio, *prio);
	}

	/*ST_ADD_0701, For frame rate setup */
	case VIDIOC_S_PARM:
	{

		struct v4l2_streamparm *sp = arg;
     		struct v4l2_captureparm *cp = &sp->parm.capture;
 		struct v4l2_fract *tpf = &cp->timeperframe;
		u32		tempvalue = 0, framerate = 0 ;

		if( cp->capturemode & V4L2_CAP_TIMEPERFRAME ) {

			if( cp->timeperframe.numerator == 0 ||cp->timeperframe.denominator == 0 ) {
				cp->timeperframe.numerator = 30 ;
				cp->timeperframe.denominator = 30 ;
			}

			DBG( "VIDIOC_S_PARM: %d,%d ==> %x", cp->timeperframe.numerator, cp->timeperframe.denominator,  tempvalue ) ;

			framerate = ( cp->timeperframe.numerator * 256 ) / cp->timeperframe.denominator ;
	
			tempvalue = ( framerate & 0xFF )  | ((framerate&0xFF)<<8) ;

			/* 0x04c */
			btwrite( tempvalue, 0x04C ) ;	
		
		}

		return 0 ;

	}



	case VIDIOC_ENUMSTD:
	case VIDIOC_G_STD:
	case VIDIOC_S_STD:
	case VIDIOC_ENUMINPUT:
	case VIDIOC_G_INPUT:
	case VIDIOC_S_INPUT:
	case VIDIOC_G_TUNER:
	case VIDIOC_S_TUNER:
	case VIDIOC_G_FREQUENCY:
	case VIDIOC_S_FREQUENCY:
	case VIDIOC_LOG_STATUS:
		
		return tw_common_ioctls(btv,cmd,arg);

	default:
		return -ENOIOCTLCMD;
	}

	// DBG( "<-----tw_do_ioctl\n" ) ;
	
	return 0;

 fh_unlock_and_return:

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
	mutex_unlock(&fh->cap.vb_lock);
#else
	mutex_unlock(&fh->cap.lock);
#endif
	// DBG( "<-----tw_do_ioctl - failed .... \n" ) ;
	
	return retval;
	
}



static int tw_ioctl(struct inode *inode, struct file *file,
		      unsigned int cmd, unsigned long arg)
{
	struct bttv_fh *fh  = file->private_data;

	switch (cmd) {
		
	case BTTV_VBISIZE:
		
		DBG( "tw_ioctl (BTV_VBISIZE) ... \n " ) ;

		tw_switch_type(fh,V4L2_BUF_TYPE_VBI_CAPTURE);
		return fh->lines * 2 * 2048;

	default:
		DBG("tw_ioctl(cmd: %x) ...\n", cmd ) ;
		return video_usercopy(inode, file, cmd, arg, tw_do_ioctl);
	}
}

static ssize_t bttv_read(struct file *file, char __user *data,
			 size_t count, loff_t *ppos)
{
	struct bttv_fh *fh = file->private_data;
	int retval = 0;

	DBG( "-----> bttv_read... \n" ) ;	

	if (fh->btv->errors) {
		DBG( "<----- bttv_read=error\n" ) ;
		bttv_reinit_tw680x(fh->btv);
	}	

	DBG("bttv%d: read count=%d type=%s\n",
		fh->btv->c.nr,(int)count,v4l2_type_names[fh->type]);

	switch (fh->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:

		DBG( "bttv_read(VIDEO)...\n" ) ;

		// ST_0725
		//if (locked_btres(fh->btv,RESOURCE_VIDEO))
		if (!check_alloc_btres(fh->btv, fh, RESOURCE_VIDEO_READ )) {
			return -EBUSY;
		}

		retval = videobuf_read_one(&fh->cap, data, count, ppos,
					   file->f_flags & O_NONBLOCK);

		// ST_0725
		free_btres(fh->btv, fh, RESOURCE_VIDEO_READ );

		break;
	case V4L2_BUF_TYPE_VBI_CAPTURE:

		DBG( "bttv_read(VBI)...\n" ) ;
		
		if (!check_alloc_btres(fh->btv,fh,RESOURCE_VBI)) {
			DBG(  "Check_alloc_btres: BUSY \n" ) ;
			return -EBUSY;
		}
			
		retval = videobuf_read_stream(&fh->vbi, data, count, ppos, 1,
					      file->f_flags & O_NONBLOCK);

		// ST_0725
		free_btres(fh->btv, fh, RESOURCE_VBI );

		// ST_TEST_0318 for VBI support 
		// ST_0725
		// retval = 1 ;
		retval = 1 ;


		break;
	default:
		BUG();
	}

	//DBG( "<----- bttv_read... \n" ) ;	


	return retval;


}

static unsigned int 
bttv_poll
(
	struct file *file, 
	poll_table *wait
)
{
	struct bttv_fh *fh = file->private_data;
	struct bttv_buffer *buf;
	enum v4l2_field field;
	struct bttv *btv = fh->btv ;


	DBG( "--->bttv_poll  - %d \n", fh->btv->c.nr ) ;
	
	// ST_0122_DEL
	/*
	if (V4L2_BUF_TYPE_VBI_CAPTURE == fh->type) {
		if (!check_alloc_btres(fh->btv,fh,RESOURCE_VBI))
			return POLLERR;
		return videobuf_poll_stream(file, &fh->vbi, wait);
	}
	*/


	if ( check_btres(fh,RESOURCE_VIDEO) ) {
		
		/* streaming capture */
		if (list_empty(&fh->cap.stream)) {
			DBG( "bttv_poll - POLLERR -%d ... \n", fh->btv->c.nr  ) ;
			return POLLERR;
		}
		
		buf = list_entry(fh->cap.stream.next,struct bttv_buffer,vb.stream);

		DBG( "bttv_poll - %d, bttv_buffer: %x \n", fh->btv->c.nr, buf  ) ;
		
	} else {
	
		/* read() capture */

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_lock(&fh->cap.vb_lock);
#else
		mutex_lock(&fh->cap.lock);
#endif

		DBG( "bttv_poll::Read Capture ... \n" ) ;

		if (NULL == fh->cap.read_buf) {

			/* need to capture a new frame */
			if (locked_btres(fh->btv,RESOURCE_VIDEO))
				goto _pool_error ;


// ST_0303_ADD
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	

			fh->cap.read_buf = videobuf_sg_alloc(fh->cap.msize);
		
#else

			fh->cap.read_buf = videobuf_alloc(fh->cap.msize);
#endif
			
			if ( NULL == fh->cap.read_buf ) 
				goto _pool_error ;				
			
			fh->cap.read_buf->memory = V4L2_MEMORY_USERPTR;

			DBG_SWITCH( "bttv_poll:: Videobuf_next_field ... \n" ) ;
			
			field = videobuf_next_field(&fh->cap);
			
			if (0 != fh->cap.ops->buf_prepare(&fh->cap,fh->cap.read_buf,field)) {
				
				kfree (fh->cap.read_buf);
				
				fh->cap.read_buf = NULL;

				goto _pool_error ;

			}

			fh->cap.ops->buf_queue(&fh->cap,fh->cap.read_buf);
			
			fh->cap.read_off = 0;

		}

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		mutex_unlock(&fh->cap.vb_lock);
#else				
		mutex_unlock(&fh->cap.lock);
#endif

		buf = (struct bttv_buffer*)fh->cap.read_buf;

	}

	DBG( "bttv_poll::poll_wait - %d, buf = %x  \n", fh->btv->c.nr, buf ) ;

	poll_wait(file, &buf->vb.done, wait);

	if (buf->vb.state == STATE_DONE || buf->vb.state == STATE_ERROR) {
		DBG( "bttv_poll - %d::state = %d !!! \n", fh->btv->c.nr, buf->vb.state ) ;
		return POLLIN|POLLRDNORM;
	}

	return 0 ;

_pool_error:

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
	mutex_unlock(&fh->cap.vb_lock);
#else				
	mutex_unlock(&fh->cap.lock);
#endif

	DBG( "bttv_poll - POLLERR - %d !!!  \n", fh->btv->c.nr ) ;
     	return POLLERR ;


}



static int 
tw_open(struct inode *inode, struct file *file)
{
	int minor = iminor(inode);
	struct bttv *btv = NULL;
	struct bttv_fh *fh;
	enum v4l2_buf_type type = 0;
	unsigned int i;

	DBG( KERN_DEBUG "tw680x: open minor=%d\n",minor);

	// minor ++ ;

	DBG( "tw_open: %d \n", minor ) ;

	lock_kernel();

	for (i = 0; i < bttv_num; i++) {
		
		if (bttvs[i].video_dev && bttvs[i].video_dev->minor == minor) {

			btv = &bttvs[i];

			type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

			/* ST_0814 */
			/*
			init_irqreg(btv);
			btor( (TW6801_VDMAC_VDMAP_EN|TW6801_VDMAC_VFIFO_EN), TW6801_VDMAC );		
			DBG_SWITCH( "VIDEO Open(i=%d), VDMAC--> %x.......\n", i, btread(TW6801_VDMAC)) ;
			*/

			break;
		}
		
	}

	
	if (NULL == btv) {
		DBG( "<----Device not found .... \n" ) ;
		return -ENODEV;
	}

	/* ST_0903, Reject if open second time */
	if( btv->users > 0 ) {
		DBG( " This device can NOT be open twice ....\n" ) ;
		return -EBUSY ;
	}


	DBG( "tw680x-%d: open called (type=%s), users: %d \n",
		btv->c.nr,v4l2_type_names[type], btv->users );

	/* allocate per filehandle data */
	fh = kmalloc(sizeof(*fh),GFP_KERNEL);

	if (NULL == fh)
		return -ENOMEM;

	file->private_data = fh;
	*fh = btv->init;
	fh->type = type;
	fh->ov.setup_ok = 0;
	v4l2_prio_open(&btv->prio,&fh->prio);


// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
	videobuf_queue_sg_init(	&fh->cap, 
						&bttv_video_qops,
			    			&btv->c.pci->dev, 
			    			&btv->s_lock,
			    			V4L2_BUF_TYPE_VIDEO_CAPTURE,
						V4L2_FIELD_INTERLACED,	
			    			sizeof(struct bttv_buffer),
			    			fh ) ;

	
	videobuf_queue_sg_init(	&fh->vbi, 
						&bttv_vbi_qops,	
			  	  		&btv->c.pci->dev, 
			  	  		&btv->s_lock,
			    			V4L2_BUF_TYPE_VBI_CAPTURE,
			    			V4L2_FIELD_SEQ_TB,
			    			sizeof(struct bttv_buffer),
			    			fh ) ;

	
#else

	videobuf_queue_init(&fh->cap, &bttv_video_qops,
			    btv->c.pci, &btv->s_lock,
			    V4L2_BUF_TYPE_VIDEO_CAPTURE,
			    V4L2_FIELD_INTERLACED,
			    sizeof(struct bttv_buffer),
			    fh);

	videobuf_queue_init(&fh->vbi, &bttv_vbi_qops,	
			    btv->c.pci, &btv->s_lock,
			    V4L2_BUF_TYPE_VBI_CAPTURE,
			    V4L2_FIELD_SEQ_TB,
			    sizeof(struct bttv_buffer),
			    fh);
	
#endif
			    
	i2c_vidiocschan(btv);

	btv->users++;
	
	if (V4L2_BUF_TYPE_VBI_CAPTURE == fh->type)
		bttv_vbi_setlines(fh,btv,16);

	bttv_field_count(btv);

	btv->m_IntCounter = 0 ;
	btv->m_TotalINTCounter = 0 ;
	btv->m_LastTimeStamp = 0 ;
	btv->m_LastTS = 0 ;
	btv->m_isfirsttime= 0 ;

	/* ST_090100, for APP try to STOP in DMAERR state */
	btv->m_stop_by_user =  0 ;
	tw_dma_init_main(btv);	

	/* ST_0902, for DMA error handle */
	btv->m_dma_error_count = 0 ; 
	/* ST_0902, reset the hw frame trop register */
	btwrite(0x0000, 0x04C ) ;			// 0x04c



	unlock_kernel();


	return 0;

}





static int 
tw_release
(
	struct inode *inode, 
	struct file *file
)
{
	struct bttv_fh *fh = file->private_data;
	struct bttv *btv = fh->btv;
	int minor = iminor(inode);
	unsigned int i;

	DBG_SWITCH( "=== tw_release: %d === \n", minor ) ;

	// ST_ADD_1122
	for (i = 0; i < bttv_num; i++) {

		if (bttvs[i].video_dev && bttvs[i].video_dev->minor == minor) {

			btv = &bttvs[i];

			/* ST_0814 */
			/*
			btwrite( 0, TW6801_INT_MASK ) ;	
			btwrite(~0x0, TW6801_INTSTAT);
			btand( ~(TW6801_VDMAC_VDMAP_EN|TW6801_VDMAC_VFIFO_EN), TW6801_VDMAC ) ;		
			*/
			tw_DisableDMA( btv ) ;
			
			DBG_SWITCH( "tw_release(i=%d), VDMAC--->%x  .......\n", i, btread(TW6801_VDMAC) ) ;

			break ;

		}

	
	}
	
	/* ST_090100, for APP try to STOP in DMAERR state */
	btv->m_stop_by_user =  1 ;

		
	/* turn off overlay */
	if (check_btres(fh, RESOURCE_OVERLAY))
		bttv_switch_overlay(btv,fh,NULL);

	/* stop video capture */
	if (check_btres(fh, RESOURCE_VIDEO)) {
		videobuf_streamoff(&fh->cap);
		free_btres(btv,fh,RESOURCE_VIDEO);
	}
	if (fh->cap.read_buf) {
		buffer_release(&fh->cap,fh->cap.read_buf);
		kfree(fh->cap.read_buf);
	}

	/* stop vbi capture */
	if( &btv->timeout )	del_timer(&btv->timeout);

	if( switching ) {
		if( &btv->RISCTimer ) del_timer( &btv->RISCTimer ) ;
	}

	// 

	/* free stuff */
	videobuf_mmap_free(&fh->cap);
	videobuf_mmap_free(&fh->vbi);
	v4l2_prio_close(&btv->prio,&fh->prio);
	file->private_data = NULL;
	kfree(fh);

	btv->users--;

	bttv_field_count(btv);

	return 0;

}




static int
bttv_mmap
(
	struct file *file, 
	struct vm_area_struct *vma
)
{
	struct bttv_fh *fh = file->private_data;

	dprintk("bttv%d: mmap type=%s 0x%lx+%ld\n",
		fh->btv->c.nr, v4l2_type_names[fh->type],
		vma->vm_start, vma->vm_end - vma->vm_start);
	
	return videobuf_mmap_mapper(bttv_queue(fh),vma);
	
}

static struct file_operations bttv_fops =
{
	.owner	  	= THIS_MODULE,
	.open	  	= tw_open,
	.release  		= tw_release,
	.ioctl	  	= tw_ioctl,
	.compat_ioctl	= v4l_compat_ioctl32,
	.llseek	  	= no_llseek,
	.read	  	= bttv_read,
	.mmap	  	= bttv_mmap,
	.poll     		= bttv_poll,
};

static struct video_device bttv_video_template =
{
	.name     = "UNSET",

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)	
	.type     = VID_TYPE_CAPTURE|VID_TYPE_TUNER|
		    	VID_TYPE_CLIPPING|VID_TYPE_SCALES,
	.hardware = VID_HARDWARE_BT848,
#endif

	
	.fops     = &bttv_fops,
	.minor    = -1,
};

static struct video_device bttv_vbi_template =
{
	.name     = "TW6801 VBI",

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)	
	.type     = VID_TYPE_TUNER|VID_TYPE_TELETEXT,
	.hardware = VID_HARDWARE_BT848,
#endif

	.fops     = &bttv_fops,
	.minor    = -1,
};

/* ----------------------------------------------------------------------- */
/* radio interface                                                         */

static int radio_open(struct inode *inode, struct file *file)
{
	int minor = iminor(inode);
	struct bttv *btv = NULL;
	unsigned int i;

	dprintk("bttv: open minor=%d\n",minor);

	for (i = 0; i < bttv_num; i++) {
		if (bttvs[i].radio_dev->minor == minor) {
			btv = &bttvs[i];
			break;
		}
	}
	if (NULL == btv)
		return -ENODEV;

	dprintk("bttv%d: open called (radio)\n",btv->c.nr);
	mutex_lock(&btv->lock);

	btv->radio_user++;

	file->private_data = btv;

	bttv_call_i2c_clients(btv,AUDC_SET_RADIO,NULL);
	audio_input(btv,TVAUDIO_INPUT_RADIO);

	mutex_unlock(&btv->lock);
	return 0;
}

static int radio_release(struct inode *inode, struct file *file)
{
	struct bttv        *btv = file->private_data;
	struct rds_command cmd;

	btv->radio_user--;

	bttv_call_i2c_clients(btv, RDS_CMD_CLOSE, &cmd);

	return 0;
}

static int radio_do_ioctl(struct inode *inode, struct file *file,
			  unsigned int cmd, void *arg)
{
	struct bttv    *btv = file->private_data;

	switch (cmd) {
	case VIDIOCGCAP:
	{
		struct video_capability *cap = arg;

		memset(cap,0,sizeof(*cap));
		strcpy(cap->name,btv->radio_dev->name);
		cap->type = VID_TYPE_TUNER;
		cap->channels = 1;
		cap->audios = 1;
		return 0;
	}

	case VIDIOCGTUNER:
	{
		struct video_tuner *v = arg;

		if(v->tuner)
			return -EINVAL;
		memset(v,0,sizeof(*v));
		strcpy(v->name, "Radio");
		bttv_call_i2c_clients(btv,cmd,v);
		return 0;
	}
	case VIDIOCSTUNER:
		/* nothing to do */
		return 0;

	case BTTV_VERSION:
	case VIDIOCGFREQ:
	case VIDIOCSFREQ:
	case VIDIOCGAUDIO:
	case VIDIOCSAUDIO:
	case VIDIOC_LOG_STATUS:
		return tw_common_ioctls(btv,cmd,arg);
	default:
		return -ENOIOCTLCMD;
	}
	return 0;
}

static int radio_ioctl(struct inode *inode, struct file *file,
		       unsigned int cmd, unsigned long arg)
{
	return video_usercopy(inode, file, cmd, arg, radio_do_ioctl);
}

static ssize_t radio_read(struct file *file, char __user *data,
			 size_t count, loff_t *ppos)
{
	struct bttv    *btv = file->private_data;
	struct rds_command cmd;
	cmd.block_count = count/3;
	cmd.buffer = data;
	cmd.instance = file;
	cmd.result = -ENODEV;

	bttv_call_i2c_clients(btv, RDS_CMD_READ, &cmd);

	return cmd.result;
}

static unsigned int radio_poll(struct file *file, poll_table *wait)
{
	struct bttv    *btv = file->private_data;
	struct rds_command cmd;
	cmd.instance = file;
	cmd.event_list = wait;
	cmd.result = -ENODEV;
	bttv_call_i2c_clients(btv, RDS_CMD_POLL, &cmd);

	return cmd.result;
}

static struct file_operations radio_fops =
{
	.owner	  = THIS_MODULE,
	.open	  = radio_open,
	.read     = radio_read,
	.release  = radio_release,
	.ioctl	  = radio_ioctl,
	.llseek	  = no_llseek,
	.poll     = radio_poll,
};

static struct video_device radio_template =
{
	.name     = "bt848/878 radio",

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)	
	.type     = VID_TYPE_TUNER,
	.hardware = VID_HARDWARE_BT848,
#endif
	.fops     = &radio_fops,
	.minor    = -1,
};

/* ----------------------------------------------------------------------- */
/* some debug code                                                         */

static int bttv_risc_decode(u32 risc)
{
	static char *instr[16] = {
		[ BT848_RISC_WRITE     >> 28 ] = "write",
		[ BT848_RISC_SKIP      >> 28 ] = "skip",
		[ BT848_RISC_WRITEC    >> 28 ] = "writec",
		[ BT848_RISC_JUMP      >> 28 ] = "jump",
		[ BT848_RISC_SYNC      >> 28 ] = "sync",
		[ BT848_RISC_WRITE123  >> 28 ] = "write123",
		[ BT848_RISC_SKIP123   >> 28 ] = "skip123",
		[ BT848_RISC_WRITE1S23 >> 28 ] = "write1s23",
	};
	static int incr[16] = {
		[ BT848_RISC_WRITE     >> 28 ] = 2,
		[ BT848_RISC_JUMP      >> 28 ] = 2,
		[ BT848_RISC_SYNC      >> 28 ] = 2,
		[ BT848_RISC_WRITE123  >> 28 ] = 5,
		[ BT848_RISC_SKIP123   >> 28 ] = 2,
		[ BT848_RISC_WRITE1S23 >> 28 ] = 3,
	};
	static char *bits[] = {
		"be0",  "be1",  "be2",  "be3/resync",
		"set0", "set1", "set2", "set3",
		"clr0", "clr1", "clr2", "clr3",
		"irq",  "res",  "eol",  "sol",
	};
	int i;

	DBG("0x%08x [ %s", risc,
	       instr[risc >> 28] ? instr[risc >> 28] : "INVALID");
	for (i = ARRAY_SIZE(bits)-1; i >= 0; i--)
		if (risc & (1 << (i + 12)))
			DBG(" %s",bits[i]);
	DBG(" count=%d ]\n", risc & 0xfff);
	return incr[risc >> 28] ? incr[risc >> 28] : 1;
}

static void bttv_risc_disasm(struct bttv *btv,
			     struct btcx_riscmem *risc)
{
	unsigned int i,j,n;

	DBG("%s: risc disasm: %p [dma=0x%08lx]\n",
	       btv->c.name, risc->cpu, (unsigned long)risc->dma);
	for (i = 0; i < (risc->size >> 2); i += n) {
		DBG("%s:   0x%lx: ", btv->c.name,
		       (unsigned long)(risc->dma + (i<<2)));
		n = bttv_risc_decode(risc->cpu[i]);
		for (j = 1; j < n; j++)
			DBG("%s:   0x%lx: 0x%08x [ arg #%d ]\n",
			       btv->c.name, (unsigned long)(risc->dma + ((i+j)<<2)),
			       risc->cpu[i+j], j);
		if (0 == risc->cpu[i])
			break;
	}
}

static void bttv_print_riscaddr(struct bttv *btv)
{
	DBG("  main: %08Lx\n",
	       (unsigned long long)btv->main.dma);
	DBG("  vbi : o=%08Lx e=%08Lx\n",
	       btv->cvbi ? (unsigned long long)btv->cvbi->top.dma : 0,
	       btv->cvbi ? (unsigned long long)btv->cvbi->bottom.dma : 0);
	DBG("  cap : o=%08Lx e=%08Lx\n",
	       btv->curr.top    ? (unsigned long long)btv->curr.top->top.dma : 0,
	       btv->curr.bottom ? (unsigned long long)btv->curr.bottom->bottom.dma : 0);
	DBG("  scr : o=%08Lx e=%08Lx\n",
	       btv->screen ? (unsigned long long)btv->screen->top.dma : 0,
	       btv->screen ? (unsigned long long)btv->screen->bottom.dma : 0);
	bttv_risc_disasm(btv, &btv->main);
}

/* ----------------------------------------------------------------------- */
/* irq handler                                                             */

static char *irq_name[] = {
	"FMTCHG",  // format change detected (525 vs. 625)
	"VSYNC",   // vertical sync (new field)
	"HSYNC",   // horizontal sync
	"OFLOW",   // chroma/luma AGC overflow
	"HLOCK",   // horizontal lock changed
	"VPRES",   // video presence changed
	"6", "7",
	"I2CDONE", // hw irc operation finished
	"GPINT",   // gpio port triggered irq
	"10",
	"RISCI",   // risc instruction triggered irq
	"FBUS",    // pixel data fifo dropped data (high pci bus latencies)
	"FTRGT",   // pixel data fifo overrun
	"FDSR",    // fifo data stream resyncronisation
	"PPERR",   // parity error (data transfer)
	"RIPERR",  // parity error (read risc instructions)
	"PABORT",  // pci abort
	"OCERR",   // risc instruction error
	"SCERR",   // syncronisation error
};

static void tw_print_irqbits(u32 print, u32 mark)
{
	unsigned int i;

	DBG("bits:");
	for (i = 0; i < ARRAY_SIZE(irq_name); i++) {
		if (print & (1 << i))
			DBG(" %s",irq_name[i]);
		if (mark & (1 << i))
			DBG("*");
	}
}

static void bttv_irq_debug_low_latency(struct bttv *btv, u32 rc)
{
	DBG("bttv%d: irq: skipped frame [main=%lx,o_vbi=%lx,o_field=%lx,rc=%lx]\n",
	       btv->c.nr,
	       (unsigned long)btv->main.dma,
	       (unsigned long)btv->main.cpu[RISC_SLOT_O_VBI+1],
	       (unsigned long)btv->main.cpu[RISC_SLOT_O_FIELD+1],
	       (unsigned long)rc);

	if (0 == (btread(BT848_DSTATUS) & BT848_DSTATUS_HLOC)) {
		DBG("bttv%d: Oh, there (temporarely?) is no input signal. "
		       "Ok, then this is harmless, don't worry ;)\n",
		       btv->c.nr);
		return;
	}
	DBG("bttv%d: Uhm. Looks like we have unusual high IRQ latencies.\n",
	       btv->c.nr);
	DBG("bttv%d: Lets try to catch the culpit red-handed ...\n",
	       btv->c.nr);
	dump_stack();
}


static int
tw_irq_next_video
(
	struct bttv *btv, 
	struct bttv_buffer_set *set
)
{
	struct bttv_buffer *item;
	int	FieldChange = 0 ;

	DBG( "----->tw_irq_next_video\n" ) ;

	memset(set,0,sizeof(*set));

	/* capture request ? */
	if ( !list_empty(&btv->capture) ) {

		DBG( "--->Capture request-%d \n", btv->c.nr ) ;

		// ST_0126
		set->frame_irq = 1;
		// set->irqflags = 1;

		item = list_entry(btv->capture.next, struct bttv_buffer, vb.queue) ;

		// ST_0121_NOTE
		// Check field type first, if has bottom then set the buffer to bottom otherwise set it to top.
		// Then get next buffer and check the filed type, if top is NULL and the field type is TOP then set the buffer to TOP
		// Otherwise set it to bottom....
		// That meanign we will get the buffer two time and try to fill TOP and BOTTOM filed...

// ST_0126
#if 0
		// ST_0407_TEST

		if ( V4L2_FIELD_HAS_BOTH(item->vb.field) ) {
			DBG_MMAP( "-->(Field: %d)TOP field - %d\n", item->vb.field, btv->c.nr ) ;
			set->top = item ;
			if( item->vb.queue.next != &btv->capture) {
				item = list_entry(item->vb.queue.next, struct bttv_buffer, vb.queue);
				if ( V4L2_FIELD_HAS_BOTH(item->vb.field) ) {
					set->bottom = item ;
					DBG_MMAP( "-->(Field: %d)Get BOTTOM field again - %d\n", item->vb.field, btv->c.nr ) ;
				} else {
					DBG_MMAP( "-->(Field: %d) It is NOT BOTTOM field !!! dev:  %d\n", item->vb.field, btv->c.nr ) ;
				}
			} else {
				DBG_MMAP( ">>>> Buffer is empty (dev->%d) <<<< \n", btv->c.nr ) ;
			}
		} else {

			if (V4L2_FIELD_HAS_TOP(item->vb.field) ){
				set->top    = item ;
				DBG_MMAP( "   -->(Field: %d)Has TOP-%d\n", item->vb.field, btv->c.nr ) ;

			}

			if (V4L2_FIELD_HAS_BOTTOM(item->vb.field) ) {
				set->bottom = item ;
				DBG_MMAP( "   -->(Field: %d)Has BOTTOM-%d\n", item->vb.field, btv->c.nr ) ;
			}
		}
#endif // ST_0407
			
#if 1
		if (V4L2_FIELD_HAS_TOP(item->vb.field) ){
			set->top    = item ;
			//if( btv->c.nr == DBG_DeviceNumber ) 
			DBG( "   -->(Field: %d)Has TOP-%d\n", item->vb.field, btv->c.nr ) ;

		} 
		
		if (V4L2_FIELD_HAS_BOTTOM(item->vb.field) ) {
			set->bottom = item ;
			//if( btv->c.nr == DBG_DeviceNumber ) 
			DBG( "   -->(Field: %d)Has BOTTOM-%d\n", item->vb.field, btv->c.nr ) ;
		}

		/*ST_DEL_072409 */
#if 1
                if (!V4L2_FIELD_HAS_BOTH(item->vb.field) && (item->vb.queue.next != &btv->capture)) {

			//if( btv->c.nr == DBG_DeviceNumber ) {
				DBG( "   -->(Field: %d) get second field -%d, TOP: %x, BOTTOM: %x\n", 
						item->vb.field, 
						btv->c.nr,
						set->top,
						set->bottom ) ;
                	//}
			
			item = list_entry(item->vb.queue.next, struct bttv_buffer, vb.queue);

                        if (!V4L2_FIELD_HAS_BOTH(item->vb.field)) {
							
                                if (NULL == set->top && V4L2_FIELD_TOP == item->vb.field) {
                                        set->top = item;
					//if( btv->c.nr == DBG_DeviceNumber ) {
						DBG_MMAP( "   -->(Field: %d) get second field-%d, TOP: %x \n", 
							item->vb.field, 
							btv->c.nr,
							set->top ) ;
					//}
                                }
								
                                if (NULL == set->bottom &&  V4L2_FIELD_BOTTOM == item->vb.field) {
                                        set->bottom = item;
					//if( btv->c.nr == DBG_DeviceNumber ) {
						DBG_MMAP( "   -->(Field: %d) get second field-%d, BOTTOM: %x \n", 
							item->vb.field, 
							btv->c.nr,
							set->bottom ) ;
					//}	
                                }

								
                                if (NULL != set->top  &&  NULL != set->bottom)
                                        set->top_irq = 2 ;
								
                	}						
			
		}

#endif

#else
		if( btv->FieldType == TW6801_DMA_TYPE_SYNCO ) {
			
			if (V4L2_FIELD_HAS_TOP(item->vb.field) ){
				set->top    = item ;
				DBG_SWITCH( "   -->(Field: %d)Has TOP-%d\n", item->vb.field, btv->c.nr ) ;
				btv->FieldType = TW6801_DMA_TYPE_SYNCE ;
			}
			
		} 
		if( btv->FieldType == TW6801_DMA_TYPE_SYNCE ) {
			if (V4L2_FIELD_HAS_BOTTOM(item->vb.field) ) {
				set->bottom = item ;
				DBG_SWITCH( "   -->(Field: %d)Has BOTTOM-%d\n", item->vb.field, btv->c.nr ) ;
				btv->FieldType = TW6801_DMA_TYPE_SYNCO ;	
			}
		}

		if( item->vb.queue.next != &btv->capture ) {
			item = list_entry(item->vb.queue.next, struct bttv_buffer, vb.queue);
			if ( set->top == NULL &&  btv->FieldType == TW6801_DMA_TYPE_SYNCO) {
				set->top = item ;	
				btv->FieldType = TW6801_DMA_TYPE_SYNCE ;					
			} else if ( set->bottom == NULL && btv->FieldType == TW6801_DMA_TYPE_SYNCE ) {
				set->bottom = item ;
				btv->FieldType = TW6801_DMA_TYPE_SYNCO ;					
			}
		} else {
			DBG_SWITCH( "   -->Buffer is empty %d\n", btv->c.nr ) ;
		}
		

		
#endif		

	} else { 
	
		// ST_11182008, Show the status if tehre are no any buffer available ...

		DBG_FPS( "Buffer is empty-Device: #%d !!!!\n", btv->c.nr	 ) ;

		/* ST_ADD_072409 */
		// return (-1) ;

	}

	/* screen overlay ? */
	if (NULL != btv->screen) {

		DBG ( "screen overlay -->\n" ) ;
		
		if (V4L2_FIELD_HAS_BOTH(btv->screen->vb.field)) {
			if (NULL == set->top && NULL == set->bottom) {
				set->top    = btv->screen;
				set->bottom = btv->screen;
			}
		} else {
			if (V4L2_FIELD_TOP == btv->screen->vb.field &&
			    NULL == set->top) {
				set->top = btv->screen;
			}
			if (V4L2_FIELD_BOTTOM == btv->screen->vb.field &&
			    NULL == set->bottom) {
				set->bottom = btv->screen;
			}
		}
	}


	DBG_SWITCH( "TW6801-%d: next set: top=%p bottom=%p [screen=%p, irqflags=%d, top_irq: %d]\n",
		btv->c.nr,set->top, set->bottom,
		btv->screen,	set->frame_irq,set->top_irq);

	DBG( "<-----tw_irq_next_video\n" ) ;
	
	return 0;
	
}



static void
tw_irq_wakeup_video
(
	struct bttv *btv, 
	struct bttv_buffer_set *wakeup,
	struct bttv_buffer_set *curr, 
	unsigned int state
)

{
	struct timeval ts ;
	unsigned long TempTS ;
	
	DBG_SWITCH( "----->tw_irq_wakeup_video-#%d\n", btv->c.nr ) ;
	
	do_gettimeofday(&ts);

	if( state == STATE_ERROR ) {
		DBG_MMAP( "----->tw_irq_wakeup_video( dev: %d ---> state = ERROR !! ( time: %d )\n", btv->c.nr, ts.tv_usec ) ;
	}


	if (wakeup->top == wakeup->bottom) {
		
		// ST_0405
		//if( btv->c.nr == DBG_DeviceNumber ){
		// DBG_SWITCH( "wakeup->top == wakeup->bottom !!! \n" ) ;	
		//}
		if (NULL != wakeup->top && curr->top != wakeup->top) {
		
			//if( btv->c.nr == DBG_DeviceNumber  ) {
			DBG("TW6801-%d: wakeup: both=%x, current: %x, ts: %d \n",btv->c.nr,wakeup->top, curr->top, ts.tv_usec );
			//}

			wakeup->top->vb.ts = ts;
			wakeup->top->vb.field_count = btv->field_count;
			wakeup->top->vb.state = state;

			// ST_0129, add for switching mode support 
			if(switching) {
				if( state == STATE_DONE ) {
					
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)					
					struct videobuf_dmabuf *dma = videobuf_to_dma(&wakeup->top->vb);
					btv->m_VirtualAddress = (unsigned char *) phys_to_virt(sg_dma_address(dma->sglist) ) ;
#else
					btv->m_VirtualAddress = (unsigned char *) phys_to_virt(sg_dma_address(wakeup->top->vb.dma.sglist) ) ;
#endif

					// ST_0508_ADD
					// *btv->m_VirtualAddress = btv->m_LastInput ;

					int	TempInputStatus = 0 ;

					if( btv->m_ChannelStatus[btv->m_LastInput] == CONNECTED ) {
						TempInputStatus = 0x10 ;
					} else {
						TempInputStatus = 0x00 ;
					}

					TempInputStatus <<= btv->m_LastInput ;

					btv->m_LastInput |=  TempInputStatus;
					
					*btv->m_VirtualAddress = btv->m_LastInput  ;


				}

			}

			// ST_0414, Show the frame rate ...
#if 1
			TempTS = (ts.tv_sec * 1000) + (ts.tv_usec/1000) ;	// ms
			if( (TempTS - btv->m_LastTimeStamp) >= 1000 ) {
				btv->m_LastTimeStamp = TempTS ;
				DBG_FPS("Dev: %d, ts: %d, FPS: %d \n",btv->c.nr, TempTS, btv->m_IntCounter );
				btv->m_IntCounter = 0 ;
			}
			//
#endif
			wake_up(&wakeup->top->vb.done) ;
			
		}

	} else {

		//if( btv->c.nr == DBG_DeviceNumber  ) {
			DBG_SWITCH( "wakeup->top != bottom !!!\n" ) ;	
		//}

		
		if (NULL != wakeup->top && curr->top != wakeup->top) {

			// if( btv->c.nr ) {
				DBG_MMAP( "TW680x-%d: wakeup: top=%x, ts: %d\n",btv->c.nr,wakeup->top, ts.tv_usec);
			//}

			wakeup->top->vb.ts = ts;
			wakeup->top->vb.field_count = btv->field_count;
			wakeup->top->vb.state = state;

			// ST_0505			
			if(switching) {
				if( state == STATE_DONE ) {
					
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)					
					struct videobuf_dmabuf *dma = videobuf_to_dma(&wakeup->top->vb);
					btv->m_VirtualAddress = (unsigned char *) phys_to_virt(sg_dma_address(dma->sglist) ) ;
#else
					btv->m_VirtualAddress = (unsigned char *) phys_to_virt(sg_dma_address(wakeup->top->vb.dma.sglist) ) ;
#endif

					// ST_0508_ADD
					// *btv->m_VirtualAddress = btv->m_LastInput ;
					int	TempInputStatus = 0 ;

					if( btv->m_ChannelStatus[btv->m_LastInput] == CONNECTED ) {
						TempInputStatus = 0x10 ;
					} else {
						TempInputStatus = 0x00 ;
					}

					TempInputStatus <<= btv->m_LastInput ;

					btv->m_LastInput |=  TempInputStatus;
					
					*btv->m_VirtualAddress = btv->m_LastInput  ;

					

				}
			}


			wake_up(&wakeup->top->vb.done);

		}

		if (NULL != wakeup->bottom && curr->bottom != wakeup->bottom) {
			//if( btv->c.nr ) {	
				DBG_MMAP("TW680x-%d: wakeup: bottom=%x, ts: %d\n",btv->c.nr,wakeup->bottom, ts.tv_usec);
			//}
			wakeup->bottom->vb.ts = ts;
			wakeup->bottom->vb.field_count = btv->field_count;
			wakeup->bottom->vb.state = state;

			// ST_0505
			if(switching) {
				if( state == STATE_DONE ) {
					
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)					
					struct videobuf_dmabuf *dma = videobuf_to_dma(&wakeup->bottom->vb);
					btv->m_VirtualAddress = (unsigned char *) phys_to_virt(sg_dma_address(dma->sglist) ) ;
#else
					btv->m_VirtualAddress = (unsigned char *) phys_to_virt(sg_dma_address(wakeup->top->vb.dma.sglist) ) ;
#endif
					
					// ST_0508_ADD
					// *btv->m_VirtualAddress = btv->m_LastInput ;
					int	TempInputStatus = 0 ;

					if( btv->m_ChannelStatus[btv->m_LastInput] == CONNECTED ) {
						TempInputStatus = 0x10 ;
					} else {
						TempInputStatus = 0x00 ;
					}

					TempInputStatus <<= btv->m_LastInput ;

					btv->m_LastInput |=  TempInputStatus;
					
					*btv->m_VirtualAddress = btv->m_LastInput  ;

				}
			}


			wake_up(&wakeup->bottom->vb.done);

		}

	}

	DBG( "<-----tw_irq_wakeup_video-#%d \n", btv->c.nr ) ;
	
}




static void
bttv_irq_wakeup_vbi(struct bttv *btv, struct bttv_buffer *wakeup,
		    unsigned int state)
{
	struct timeval ts;

	if (NULL == wakeup)
		return;

	do_gettimeofday(&ts);
	wakeup->vb.ts = ts;
	wakeup->vb.field_count = btv->field_count;
	wakeup->vb.state = state;
	wake_up(&wakeup->vb.done);
}


/* ST_0902, for clean the buffer queue when stream off */
static void
tw_clear_buffer_state
(
	struct	bttv *btv
)
{
	struct bttv_buffer_set old,new;
	struct bttv_buffer *ovbi;
	struct bttv_buffer *item;
	unsigned long flags;

	struct timeval ts ;

	do_gettimeofday(&ts);

	DBG_FPS( "----->tw_clear_buffer_state - %d, ts: %d \n", btv->c.nr, ( (ts.tv_sec * 1000) + (ts.tv_usec/1000)) ) ;

	spin_lock_irqsave(&btv->s_lock,flags);

	/* deactivate stuff */
	memset(&new,0,sizeof(new));
	
	old  = btv->curr;
	ovbi = btv->cvbi;
	btv->curr = new;
	btv->cvbi = NULL;

	btv->loop_irq = 0;

	/* wake up */

	tw_irq_wakeup_video(btv, &old, &new, STATE_ERROR);

	/* cancel all outstanding capture / vbi requests */
	while (!list_empty(&btv->capture)) {
		item = list_entry(btv->capture.next, struct bttv_buffer, vb.queue);
		list_del(&item->vb.queue);
		item->vb.state = STATE_ERROR;
		wake_up(&item->vb.done);
	}

	
	while (!list_empty(&btv->vcapture)) {
		item = list_entry(btv->vcapture.next, struct bttv_buffer, vb.queue);
		list_del(&item->vb.queue);
		item->vb.state = STATE_ERROR;
		wake_up(&item->vb.done);
	}

	btv->errors++;

	DBG_FPS( "<-----tw_clear_buffer_state - %d\n", btv->c.nr ) ;
	
	spin_unlock_irqrestore(&btv->s_lock,flags);


}





static void 
tw_irq_timeout
(
	unsigned long data
)
{
	struct bttv *btv = (struct bttv *)data;
	struct bttv_buffer_set old,new;
	struct bttv_buffer *ovbi;
	struct bttv_buffer *item;
	unsigned long flags;

	struct timeval ts ;

	do_gettimeofday(&ts);


	DBG_FPS( "----->tw_irq_timeout - %d, ts: %d \n", btv->c.nr, ( (ts.tv_sec * 1000) + (ts.tv_usec/1000)) ) ;

	if (bttv_verbose) {

		// ST_DEL_0925
		/*
		printk(KERN_INFO "TW6801-%d: timeout: drop=%d irq=%d/%d, risc=%08x, ",
		       btv->c.nr, btv->framedrop, btv->irq_me, btv->irq_total,
		       btread(BT848_RISC_COUNT));
		*/		       

		DBG(KERN_INFO "TW6801-%d: timeout: drop=%d irq=%d/%d, risc=%08x \n",
		       btv->c.nr, btv->framedrop, btv->irq_me, btv->irq_total,
		       btread( TW6801_VDMAP_EXE ));


		// ST_MDIFY_0924 for tw6801 support
		// ST_DEL_1001
		// tw_print_irqbits( btread(TW6801_INTSTAT),0 ) ;
		
		//printk("\n");

	}

	spin_lock_irqsave(&btv->s_lock,flags);

	/* deactivate stuff */
	memset(&new,0,sizeof(new));
	
	old  = btv->curr;
	ovbi = btv->cvbi;
	btv->curr = new;
	btv->cvbi = NULL;

	btv->loop_irq = 0;

	tw_EnableDMA( btv ) ;
	
	/* wake up */

	// ST_0618_TEST
	tw_irq_wakeup_video(btv, &old, &new, STATE_ERROR);
	//tw_irq_wakeup_video(btv, &old, &new, STATE_NEEDS_INIT ) ;


	// ST_0127_DEL	
	// bttv_irq_wakeup_vbi(btv, ovbi, STATE_ERROR);

	/* cancel all outstanding capture / vbi requests */
	while (!list_empty(&btv->capture)) {
		item = list_entry(btv->capture.next, struct bttv_buffer, vb.queue);
		list_del(&item->vb.queue);
		item->vb.state = STATE_ERROR;
		wake_up(&item->vb.done);
	}

	
	// ST_0618_DEL
	while (!list_empty(&btv->vcapture)) {
		item = list_entry(btv->vcapture.next, struct bttv_buffer, vb.queue);
		list_del(&item->vb.queue);
		item->vb.state = STATE_ERROR;
		wake_up(&item->vb.done);
	}

	btv->errors++;

	DBG_FPS( "<-----tw_irq_timeout - %d\n", btv->c.nr ) ;
	
	spin_unlock_irqrestore(&btv->s_lock,flags);
	
}



static void
bttv_irq_wakeup_top
(
	struct bttv *btv
)
{
	struct bttv_buffer *wakeup = btv->curr.top;

	if (NULL == wakeup)
		return;

	DBG_MMAP( "---->bttv_irq_wakeup_top\n" ) ;

	spin_lock(&btv->s_lock);
	
	btv->curr.top_irq = 0;
	btv->curr.top = NULL;

	tw_dma_hook(btv, RISC_SLOT_O_FIELD, NULL, 0);

	// ST_0126_DEL
	// tw_dma_hook(btv, RISC_SLOT_E_FIELD, NULL, 0);

	do_gettimeofday(&wakeup->vb.ts);
	wakeup->vb.field_count = btv->field_count;

	wakeup->vb.state = STATE_DONE;

	wake_up(&wakeup->vb.done);
	spin_unlock(&btv->s_lock);

}



static void
bttv_irq_wakeup_bottom
(
	struct bttv *btv
)
{
	struct bttv_buffer *wakeup = btv->curr.top;

	if (NULL == wakeup)
		return;

	DBG_SWITCH( "---->bttv_irq_wakeup_top\n" ) ;

	spin_lock(&btv->s_lock);
	btv->curr.top_irq = 0;
	btv->curr.top = NULL;

	tw_dma_hook(btv, RISC_SLOT_O_FIELD, NULL, 0);
	tw_dma_hook(btv, RISC_SLOT_E_FIELD, NULL, 0);

	do_gettimeofday(&wakeup->vb.ts);
	wakeup->vb.field_count = btv->field_count;

	wakeup->vb.state = STATE_DONE;

	wake_up(&wakeup->vb.done);
	spin_unlock(&btv->s_lock);

}



static inline int is_active
(
	struct btcx_riscmem *risc, 
	u32 rc
)
{

	if (rc < risc->dma)
		return 0;
	
	if (rc > risc->dma + risc->size)
		return 0;

	DBG_FPS( "----->is_active[dma: %x, size: %d, rc: %x]\n", risc->dma, risc->size, rc ) ;

	DBG( "<-----is_active\n") ;
	
	return 1;
}

static void
tw_irq_switch_video
(
	struct bttv *btv
)
{
	struct bttv_buffer_set new;
	struct bttv_buffer_set old;
	dma_addr_t rc;

	DBG_SWITCH( "----->tw_irq_switch_video-%d\n", btv->c.nr ) ;

	// ST_0410
	// ST_0423
	//spin_lock(&btv->s_lock);

	/* ST_0812, move from bottom */
	memset( &new,0,sizeof(&new));	

	/* ST_0902, move this down */
	// tw_irq_wakeup_video( btv, &btv->curr, &new, STATE_DONE ) ;


	if(switching) {
		tw_SwitchingChannel( btv ) ;
	}

	/* ST_MOD_072909 */
#if 0

	// rc = btread( TW6801_VDMAP_EXE ) ;	
	rc = btread( TW6801_VDMAP_PP ) ;	


	if ((btv->curr.top    && is_active(&btv->curr.top->top,       rc)) ||
	    (btv->curr.bottom && is_active(&btv->curr.bottom->bottom, rc))) {
	    
		btv->framedrop++;

		if (debug_latency)
			bttv_irq_debug_low_latency(btv, rc);
	
		DBG_FPS( "===> tw_irq_switch_video-FrameDrop - top: %x, bottom: %x \n", 
				btv->curr.top->top,
				btv->curr.bottom->bottom ) ;

		// tw_buffer_reactive_video( btv, &btv->curr) ;
		/*St_072709 */
		tw_EnableDMA( btv ) ;

		return ;
		
	}
	
#endif 


	/* new buffer set */
	/* ST_ADD_072909 */
	tw_irq_next_video( btv, &new ) ;

#if 0
	if( tw_irq_next_video( btv, &new ) < 0 ) {
		DBG_MMAP( " There are no any buffer avaliable - %d .... \n", btv->c.nr ) ;
		/*St_072709 */
		tw_EnableDMA( btv ) ;
		return ;
	}
#endif


	// ST_0408_ADD, Move from bottom
#if 1

	tw_buffer_activate_video(btv, &new);

#endif	//St_0408	


	// ST_0622_DEL, move up, if happen then use old start address 
#if 0
	rc = btread( TW6801_VDMAP_EXE ) ;	

	if ((btv->curr.top    && is_active(&btv->curr.top->top,       rc)) ||
	    (btv->curr.bottom && is_active(&btv->curr.bottom->bottom, rc))) {
	    
		btv->framedrop++;

		if (debug_latency)
			bttv_irq_debug_low_latency(btv, rc);

		// ST_0410
		// spin_unlock(&btv->s_lock);
		
		DBG_MMAP( "=== tw_irq_switch_video-FrameDrop - %d === \n", btv->c.nr ) ;

		// ST_0622_DEL 
		// return ;
		
	}
	
#endif // ST_0622
	

	// ST_0408_DEL, move to up
#if 0 

	/* switch over */
	old = btv->curr;
	btv->curr = new ;

	// ST_0126
	btv->loop_irq &= ~1;
	//

	tw_buffer_activate_video(btv, &new);

#else
	old = btv->curr;
	btv->curr = new ;
	btv->loop_irq &= ~1;
	
#endif // ST_0408


	/* switch input */
	// ST_0427_DEL
	if (UNSET != btv->new_input) {
		video_mux(btv,btv->new_input);
		btv->new_input = UNSET;
	}

	/* wake up finished buffers */
	/* ST_0812, move up for test */
	/* ST_0902, move this back here */
	tw_irq_wakeup_video(btv, &old, &new, STATE_DONE);

	// ST_0410
	// ST_0423
	//spin_unlock(&btv->s_lock);

	DBG_SWITCH( "<-----tw_irq_switch_video-%d\n",  btv->c.nr ) ;
	
}





static void
bttv_irq_switch_vbi(struct bttv *btv)
{
	struct bttv_buffer *new = NULL;
	struct bttv_buffer *old;
	u32 rc;


	DBG( "----->bttv_irq_switch_vbi \n" ) ;


	spin_lock(&btv->s_lock);

	if (!list_empty(&btv->vcapture))
		new = list_entry(btv->vcapture.next, struct bttv_buffer, vb.queue);
	old = btv->cvbi;


	// ST_MODIFY_0925 for DMA control 
	// rc = btread(BT848_RISC_COUNT);
	rc = btread( TW6801_VDMAP_EXE ) ;	
	
	if (NULL != old && (is_active(&old->top,    rc) ||
			    is_active(&old->bottom, rc))) {
		btv->framedrop++;
		if (debug_latency)
			bttv_irq_debug_low_latency(btv, rc);
		spin_unlock(&btv->s_lock);
		return;
	}

	/* switch */
	btv->cvbi = new;

	bttv_buffer_activate_vbi(btv, new);

	tw_EnableDMA( btv ) ;

	bttv_irq_wakeup_vbi(btv, old, STATE_DONE);
	
	spin_unlock(&btv->s_lock);

	
}


static irqreturn_t 
tw_isr
(
	int irq, 
	void *dev_id 
)
{
	
	struct bttv *btv;
	int handled = 0;
	unsigned long flags;
	unsigned long Timeout ;
	unsigned long LoseSignalTimer  = 0 ;


	//DBG( "----->tw_isr \n" ) ;		

	btv = (struct bttv *)dev_id;

	// spin_lock_irqsave( &btv->s_lock, flags ) ;

	if (btv->custom_irq)
		handled = btv->custom_irq(btv);

// ST_0417	
// ST_0618
//	while (1) {
		
		/* get/clear interrupt status bits */
		btv->IntStatus = btread( TW6801_INTSTAT );

		/* ST_0813 */
		if( btv->IntStatus == 0x00 ) goto __tw_isr_done ;

		btv->ActiveStatus = btv->IntStatus&btread( TW6801_INT_MASK);

		handled = 1;

		DBG( "INT----->%x\n", btv->IntStatus ) ;

		btwrite( btv->IntStatus, TW6801_INTSTAT);		

		if ( !btv->ActiveStatus ) {
			// DBG( "No any IRQ needs to server ...\n" ) ;
			goto __tw_isr_done;
		}

		/* ST_DEL_072909*/
		// spin_lock_irqsave(&tw_irq_lock, flags);

		// ST_0410	
		// handled = 1;
		// btwrite( btv->IntStatus, TW6801_INTSTAT);		

		btv->DecoderStatus = btread( TW6801_DSTATUS ) ;
		btwrite( btv->DecoderStatus, TW6801_DSTATUS ) ;

		// DBG( "    ---> stat: %x, dstat: %x \n", stat, dstat );

		if ( ( btv->ActiveStatus & TW6801_GPINT) && btv->remote ) {
			DBG( "Get Remote Controller Event .....\n" ) ;
			bttv_input_irq(btv);
		}

		/* ST_0202, for tw6805 drop frame issue */
		if( btv->ActiveStatus & TW6801_FFOF ) {
			DBG_FPS( "----->TW6801_FFOF \n" ) ;

			tw_DisableDMA( btv ) ;

			tw_EnableDMA( btv ) ;
		
		}
		/* ST_0202_END */
		

		if( btv->ActiveStatus & TW6801_DMAERR ) {
			struct timeval ts;

			do_gettimeofday(&ts);
			DBG_FPS( "----->DMAERROR -%d, ts: %d \n", btv->c.nr, ( (ts.tv_sec * 1000) + (ts.tv_usec/1000)) ) ;

			/* ST_0817, re-initialize TW680x but do not set the buffer state to ERROR */
			tw_DisableDMA( btv ) ;
			
			mod_timer(&btv->timeout, jiffies+TW6801_TIMEOUT) ;		
		
			/* ST_0902, Reset the DMA table in this case */
			tw_dma_init_main(btv);	
			DBG( "Reset the start address: %x...\n", btv->main.dma ) ;
			btwrite( btv->main.dma, TW6801_VDMAP_SA ) ;			
			btv->m_dma_error_count ++ ;

			/* ST_090100, for APP try to STOP in DMAERR state */
			if( btv->m_stop_by_user ==  0 || btv->m_dma_error_count < 50 )  {
				tw_EnableDMA( btv ) ;
			} else {

				DBG( "There are a lot of DMA ERRORs, disable the DMA engine and wait the time out ....\n" ) ;

			}

		}


	
		if ( btv->ActiveStatus & TW6801_VDMAPI ) {

			struct timeval ts;
			unsigned long 	CurrentTime ;
			unsigned long 	DeltaTime ;	
	
			btv->m_IntCounter ++ ;

			do_gettimeofday(&ts);
			DBG( "----->VDMAPI(%x) -%d, ts: %d \n", btv->ActiveStatus, btv->c.nr, ( (ts.tv_sec * 1000) + (ts.tv_usec/1000)) ) ;

			/* ST_0902, for DMA error handle */
			btv->m_dma_error_count = 0 ;

/*ST_DEL_072909 */
#if 0			
			do_gettimeofday(&ts);

			CurrentTime = (ts.tv_sec * 1000) + (ts.tv_usec/1000) ;

			DeltaTime = CurrentTime - btv->m_LastTS ;

			btv->m_LastTS = CurrentTime ;

			if( DeltaTime < 2 ) {
				DBG_MMAP ( "============= DELTA TIME is too SMALL ============= \n" ) ;
			}


			// ST_0414, For INT counter 
			btv->m_IntCounter ++ ;

			// ST_0423
			btv->m_TotalINTCounter ++ ;

			// ST_0413
			//if( btv->c.nr == DBG_DeviceNumber ) {
			DBG_MMAP ( "===== TW6801_VDMAPI(btv: %x : #%d, int: %d, delta-time: %d) ===== \n", 
					btv, btv->c.nr, 
					btv->m_TotalINTCounter,
					DeltaTime  ) ;
			//}
#endif

			// ST_0410, stop DMA here then enable in buffer_queue()
			// AP will call QBUF then buffer_prepare() and buffer_queue() will be call 
			// that meaning driver have another buffer can use ...

			tw_DisableDMA( btv ) ;


/*ST_DEl_072909 */
			/* ST_ADD_073009 */
			// spin_lock_irqsave( &tw_irq_lock, flags ) ;
			
			if(switching) {

				LoseSignalTimer = tw_irq_check_channel_status( btv  ) ;
	 			tw_irq_switch_video(btv);	

				Timeout = jiffies + ( msecs_to_jiffies( LoseSignalTimer) ) ;

				btv->RISCTimer.expires = LoseSignalTimer ;
				
				mod_timer( &btv->RISCTimer, Timeout ) ;	

				DBG_SWITCH( "Set EnableRISC[%d] to 1 ..... \n", btv->c.nr ) ;				

				btv->m_EnableRISC = 1 ;	

			} else {

				/*ST_0727 */
				// spin_lock_irqsave( &tw_irq_lock, flags ) ;

				tw_irq_switch_video(btv) ;

				mod_timer(&btv->timeout, jiffies+TW6801_TIMEOUT) ;		

				tw_EnableDMA( btv ) ;

				/*ST_0727 */
				// spin_unlock_irqrestore( &tw_irq_lock, flags ) ;

			}
		
			/* ST_ADD_072909 */
			DBG( "<---- tw_isr: VDMAPI: #%d \n", btv->c.nr ) ;
			// spin_unlock_irqrestore( &tw_irq_lock, flags ) ;
		
		}


	
//	} // ST_0618

__tw_isr_done:

	btv->irq_total++;

	if (handled) btv->irq_me++;

	return IRQ_RETVAL(handled);
	
}


/* ----------------------------------------------------------------------- */
/* initialitation                                                          */

static struct video_device *vdev_init(struct bttv *btv,
				      struct video_device *template,
				      char *type)
{
	struct video_device *vfd;

	vfd = video_device_alloc();
	if (NULL == vfd)
		return NULL;
	*vfd = *template;
	vfd->minor   = -1;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
	vfd->parent	= &btv->c.pci->dev;
#else	
	vfd->dev     	= &btv->c.pci->dev;
#endif

	vfd->release = video_device_release;

	// ST_0110 for VBI support 
	/*
	snprintf(vfd->name, sizeof(vfd->name), "BT%d%s %s (%s)",
		 btv->id, (btv->id==848 && btv->revision==0x12) ? "A" : "",
		 type, bttv_tvcards[btv->c.type].name);
	*/
	
	snprintf(vfd->name, sizeof(vfd->name), "TW6801 TV Card"  );


	return vfd;
	
}

static void bttv_unregister_video(struct bttv *btv)
{
	if (btv->video_dev) {
		if (-1 != btv->video_dev->minor)
			video_unregister_device(btv->video_dev);
		else
			video_device_release(btv->video_dev);
		btv->video_dev = NULL;
	}
	if (btv->vbi_dev) {
		if (-1 != btv->vbi_dev->minor)
			video_unregister_device(btv->vbi_dev);
		else
			video_device_release(btv->vbi_dev);
		btv->vbi_dev = NULL;
	}
	if (btv->radio_dev) {
		if (-1 != btv->radio_dev->minor)
			video_unregister_device(btv->radio_dev);
		else
			video_device_release(btv->radio_dev);
		btv->radio_dev = NULL;
	}
}



/* register video4linux devices */
static int __devinit bttv_register_video(struct bttv *btv)
{

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	

	if (no_overlay > 0)
		DBG("bttv: Overlay support disabled.\n");

#else
	if (no_overlay <= 0) {
		bttv_video_template.type |= VID_TYPE_OVERLAY;
	} else {
		DBG("bttv: Overlay support disabled.\n");
	}
#endif

	/* video */
	btv->video_dev = vdev_init(btv, &bttv_video_template, "video");
	if (NULL == btv->video_dev)
		goto err;
	if (video_register_device(btv->video_dev,VFL_TYPE_GRABBER,video_nr)<0)
		goto err;

	DBG("TW6801-%d: registered device video%d\n",
	       btv->c.nr,btv->video_dev->minor & 0x1f);

	// ST_0115
// ST_0115 for 2.6.24 kernel
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)
	if (device_create_file(&btv->video_dev->dev,&dev_attr_card)<0) {
#else
	if (class_device_create_file(&btv->video_dev->dev,&class_device_attr_card)<0) {
#endif
		DBG(KERN_ERR "bttv%d: class_device_create_file 'card' failed\n", btv->c.nr);
		goto err;
	}
	

	/* vbi */
	// ST_0110 for VBI support
	// ST_0302_DEL
#if 0	
	btv->vbi_dev = vdev_init(btv, &bttv_vbi_template, "vbi");

	if (NULL == btv->vbi_dev){
		DBG( "vdev_int(VBI) Init error ...\n" ) ;
		goto err;
	}
	
	
	if (video_register_device(btv->vbi_dev,VFL_TYPE_VBI,vbi_nr)<0) {
		DBG( "<---VBI register error ...\n" ) ;
		goto err;
	}	

	DBG( "TW6801-%d: registered device vbi%d\n",
	       btv->c.nr,btv->vbi_dev->minor & 0x1f);

#endif	

	if (!btv->has_radio)
		return 0;

	/* radio */
	// ST_DEL_0310 
	/*
	btv->radio_dev = vdev_init(btv, &radio_template, "radio");

	if (NULL == btv->radio_dev)
		goto err;

	if (video_register_device(btv->radio_dev, VFL_TYPE_RADIO,radio_nr)<0)
		goto err;

	DBG(KERN_INFO "bttv%d: registered device radio%d\n",
	       btv->c.nr,btv->radio_dev->minor & 0x1f);

	*/
	/* all done */
	return 0;

 err:
	bttv_unregister_video(btv);
	return -1;
}


/* on OpenFirmware machines (PowerMac at least), PCI memory cycle */
/* response on cards with no firmware is not enabled by OF */
static void pci_set_command(struct pci_dev *dev)
{
#if defined(__powerpc__)
	unsigned int cmd;

	pci_read_config_dword(dev, PCI_COMMAND, &cmd);
	cmd = (cmd | PCI_COMMAND_MEMORY );
	pci_write_config_dword(dev, PCI_COMMAND, cmd);
#endif
}

static int __devinit tw_probe(struct pci_dev *dev,
				const struct pci_device_id *pci_id)
{
	int result;
	unsigned char lat;
	struct bttv *btv;

	int i ;

	
	DBG(KERN_INFO "----->tw_probe" );

	if (bttv_num == BTTV_MAX) return -ENOMEM;

		
	DBG(KERN_INFO "TW680x: Techwell PCI card found (%d).\n", bttv_num);
	
	btv=&bttvs[bttv_num];
	memset(btv,0,sizeof(*btv));
	btv->c.nr  = bttv_num;
	
	sprintf( btv->c.name,"bttv%d", btv->c.nr );

	/* initialize structs / fill in defaults */
	mutex_init(&btv->lock);
	mutex_init(&btv->reslock);
	spin_lock_init(&btv->s_lock);
	spin_lock_init(&btv->gpio_lock);

	// ST_0507_TEST
	spin_lock_init(&tw_irq_lock);
	//	
	
	init_waitqueue_head(&btv->gpioq);
	init_waitqueue_head(&btv->i2c_queue);
	INIT_LIST_HEAD(&btv->c.subs);
	INIT_LIST_HEAD(&btv->capture);
	INIT_LIST_HEAD(&btv->vcapture);
	v4l2_prio_init(&btv->prio);

	init_timer(&btv->timeout);
	
	btv->timeout.function 	= tw_irq_timeout;
	btv->timeout.data     	= (unsigned long)btv;

	/*ST_0807 */
	btv->timeout.expires	= jiffies + (1500000*HZ)  ;
	add_timer( &btv->timeout ) ;	
	


	// ST_0623_ADD
	//btv->timeout.expires	= jiffies + ( TW6801_TIMEOUT ) ;
	//add_timer( &btv->timeout ) ;	
	//

	btv->i2c_rc = -1;
	btv->tuner_type  = UNSET;
	btv->new_input   = UNSET;
	btv->has_radio=radio[btv->c.nr];

	/* pci stuff (init, get irq/mmio, ... */
	btv->c.pci = dev;
	btv->id  = dev->device;
	if (pci_enable_device(dev)) {
		
		DBG(KERN_WARNING "bttv%d: Can't enable device.\n", btv->c.nr);
		DBG(KERN_INFO "<-----tw_probe-1" );		
		return -EIO;
	}
	
	if (pci_set_dma_mask(dev, DMA_32BIT_MASK)) {
		DBG(KERN_WARNING "bttv%d: No suitable DMA available.\n", btv->c.nr);
		DBG(KERN_INFO "<-----tw_probe-2" );				
		return -EIO;
	}

	if ( !request_mem_region(pci_resource_start(dev,0),
				pci_resource_len(dev,0),
				btv->c.name)) {
				
		DBG(KERN_WARNING "bttv%d: can't request iomem (0x%llx).\n",
		       btv->c.nr,
		       (unsigned long long)pci_resource_start(dev,0));
		
		DBG(KERN_INFO "<-----tw_probe-3" );
		
		return -EBUSY;
	}

	// ST_0116_ADD, for 2.6.28
	pci_set_master(dev);
	pci_set_command(dev);
	pci_set_drvdata(dev,btv);
	//

	pci_read_config_byte(dev, TW6801_PCI_CLASS_REVISION, &btv->revision);
	pci_read_config_byte(dev, TW6801_PCI_LATENCY_TIMER, &lat);

	DBG(KERN_INFO "TW680x-%d (rev %d) at %s, ",
	       	btv->c.nr, btv->revision, pci_name(dev));
	
	DBG("irq: %d, latency: %d, mmio: 0x%llx\n",
	       btv->c.pci->irq, lat,
	       (unsigned long long)pci_resource_start(dev,0));

	/* ST_ADD_072909 */
	/* ST_0129 */
	// lat = 0xFF ;
	lat = 0xFF ;

	pci_write_config_byte(dev, TW6801_PCI_LATENCY_TIMER, lat);

	schedule() ;

	btv->bt848_mmio = ioremap(pci_resource_start(dev,0), 0x1000);	

	if (NULL ==btv->bt848_mmio /*ioremap(pci_resource_start(dev,0), 0x1000) */) {
		DBG_SWITCH("TW680x-%d: ioremap() failed\n", btv->c.nr);
		result = -EIO;
		DBG(KERN_INFO "<-----tw_probe-4" );				
		goto fail1;
	}

	/* identify card */
	tw_idcard(btv);

	/* disable irqs, register irq handler */
	// ST_MODIFY_0923 for disable TW6801 interrupt
	// btwrite(0, BT848_INT_MASK);
	btwrite( 0, TW6801_INT_MASK ) ;	
	
	result = request_irq(	btv->c.pci->irq, tw_isr,
			     		IRQF_SHARED /*| IRQF_DISABLED*/,btv->c.name,(void *)btv);

	if (result < 0) {
		DBG(KERN_ERR "TW6801%d: can't get IRQ %d\n",
		       bttv_num,btv->c.pci->irq);
		DBG(KERN_INFO "<-----tw_probe-5" );				
		goto fail1;
	}

	if (0 != tw_handle_chipset(btv)) {
		result = -EIO;
		DBG(KERN_INFO "<-----tw_probe-6" );				
		goto fail2;
	}

	DBG(KERN_INFO "===  tw_probe check point #1 ===\n" );				


	/* init options from insmod args */
	btv->opt_combfilter = combfilter;
	btv->opt_lumafilter = lumafilter;
	btv->opt_automute   = automute;
	btv->opt_chroma_agc = chroma_agc;
	btv->opt_adc_crush  = adc_crush;
	btv->opt_vcr_hack   = vcr_hack;
	btv->opt_whitecrush_upper  = whitecrush_upper;
	btv->opt_whitecrush_lower  = whitecrush_lower;
	btv->opt_uv_ratio   = uv_ratio;
	btv->opt_full_luma_range   = full_luma_range;
	btv->opt_coring     = coring;

	/* fill struct bttv with some useful defaults */
	btv->init.btv         		= btv;
	btv->init.ov.w.width  	= 320 ;
	btv->init.ov.w.height 	= 240 ;
	btv->init.fmt         		= format_by_palette(VIDEO_PALETTE_RGB24);
	btv->init.width      		= 320 ;
	btv->init.height     		= 240 ;
	btv->init.lines       		= 16 ;
	btv->input 			= 0 ;
	btv->FieldType = TW6801_DMA_TYPE_SYNCO ;	
	//

	/* initialize hardware */
	if (bttv_gpio)
		tw_gpio_tracking(btv,"pre-init") ;


	if(switching) {

		btv->m_CurrentChannel  = 0 ;

		for( i = 0; i < MAX_SWITCHING_NUMBER; i ++ ) {
			btv->m_Brightness[i] = BrightnessDefault ;
			btv->m_Contrast[i] = ContrastDefault ;
			btv->m_Hue[i] = HueDefault ;		
			btv->m_Saturation[i] = SaturationDefault ;
			btv->m_isDisableChannel[i] = CHANNEL_ENABLE ;		
		}

	}

	tw_dma_init_main(btv);	
	
	init_tw680x(btv);

	btwrite( 0x00, TW6801_GPOE ) ;
	
	
	if (bttv_verbose)
		tw_gpio_tracking(btv,"init");

	/* needs to be done before i2c is registered */
	tw_init_card1(btv);

	init_tw_i2c(btv);  //Denio 0125

	/* register video4linux + input */
	if (!bttv_tvcards[btv->c.type].no_video) {
		DBG( "TW680x-%d: Register video4linux + input \n", btv->c.nr ) ;
		bttv_register_video(btv);
		TW6801_bright(btv, 0x00 ) ;
		TW6801_contrast(btv, 84 );
		TW6801_hue(btv, 0 ) ;
		TW6801_sat(btv, 0x7F );
	}

	/* add subdevices */
	if (bttv_tvcards[btv->c.type].has_dvb) {


		DBG( "TW6801-%d: Add subdevice(DVB) \n", btv->c.nr ) ;

		bttv_sub_add_device(&btv->c, "dvb");
	}

	tw_input_init(btv);

	if( switching ) {
			
		init_timer( &btv->RISCTimer ) ;
		btv->RISCTimer.function 	= TimerProcess ;
		btv->RISCTimer.data 	= (unsigned long ) btv ;
		btv->RISCTimer.expires	= jiffies + ( TW6801_RISC_TIMEOUT ) ;
		btv->m_EnableRISC		= 0 ;		// Disbale RISC DMA ... 		
		add_timer( &btv->RISCTimer ) ;	

	} else {
		btv->m_EnableRISC		= 0 ;		// Disbale RISC DMA ... 		
	}


	bttv_num++;
	
	DBG(KERN_INFO "<-----tw_probe\n" );
	
	return 0;

 fail2:

	free_irq(btv->c.pci->irq,btv);
 fail1:

	if (btv->bt848_mmio)
		iounmap(btv->bt848_mmio);

	release_mem_region(pci_resource_start(btv->c.pci,0),  pci_resource_len(btv->c.pci,0)) ;

	pci_set_drvdata( dev, NULL ) ;

	return result;

}




static void __devexit 
bttv_remove
(
	struct pci_dev *pci_dev
)
{
	int i ;

	struct bttv *btv = pci_get_drvdata(pci_dev);

	if (bttv_verbose)
		DBG_SWITCH("bttv%d: unloading\n",btv->c.nr);

	// ST_ADD 
	// Try to clear the INTMASK register
	DBG_SWITCH("Clear the INTMASK register ...\n") ;
	
	btwrite( 0, TW6801_INT_MASK ) ;	
	btwrite(~0x0, TW6801_INTSTAT);

	if( &btv->timeout ) del_timer( &btv->timeout ) ;
	
	if( switching ) {
		btv->m_EnableRISC = 0 ;
		del_timer( &btv->RISCTimer ) ;
	}


	if (bttv_gpio)
		tw_gpio_tracking(btv,"cleanup");

	/* tell gpio modules we are leaving ... */
	btv->shutdown=1;
	wake_up(&btv->gpioq);
	bttv_input_fini(btv);
	bttv_sub_del_devices(&btv->c);

	/* unregister i2c_bus + input */
	fini_bttv_i2c(btv);

	/* unregister video4linux */
	bttv_unregister_video(btv);

	/* free allocated memory */
	btcx_riscmem_free(btv->c.pci,&btv->main);

	/* free ressources */
	free_irq( btv->c.pci->irq, btv ) ;
	iounmap(btv->bt848_mmio);
	release_mem_region(	pci_resource_start(btv->c.pci,0),
			   			pci_resource_len(btv->c.pci,0) ) ;

	pci_set_drvdata(pci_dev, NULL);

	return;
	
}

static int bttv_suspend(struct pci_dev *pci_dev, pm_message_t state)
{
	struct bttv *btv = pci_get_drvdata(pci_dev);
	struct bttv_buffer_set idle;
	unsigned long flags;

	DBG_SWITCH( "----->tw_suspend\n" ) ;
	
	DBG("TW6801-%d: suspend %d\n", btv->c.nr, state.event);

	/* stop dma + irqs */
	spin_lock_irqsave(&btv->s_lock,flags);
	memset(&idle, 0, sizeof(idle));
	btv->state.video = btv->curr;
	btv->state.vbi   = btv->cvbi;
	btv->curr = idle;

	// ST_0725
	// btv->state.loop_irq = btv->loop_irq;
	// btv->loop_irq = 0;

	tw_buffer_activate_video(btv, &idle);

	// ST_0622
	bttv_buffer_activate_vbi(btv, NULL);

	tw_DisableDMA( btv ) ;

	// btwrite(0, BT848_INT_MASK);
	btwrite( 0, TW6801_INT_MASK ) ;	
	btwrite(~0x0, TW6801_INTSTAT);

	
	spin_unlock_irqrestore(&btv->s_lock,flags);

	/* save bt878 state */
	btv->state.gpio_enable = btread(BT848_GPIO_OUT_EN);
	btv->state.gpio_data   = gpio_read();

	/* save pci state */
	pci_save_state(pci_dev);
	if (0 != pci_set_power_state(pci_dev, pci_choose_state(pci_dev, state))) {
		pci_disable_device(pci_dev);
		btv->state.disabled = 1;
	}

	DBG( "<-----tw_suspend\n" ) ;
	
	return 0;
}

static int bttv_resume(struct pci_dev *pci_dev)
{
	struct bttv *btv = pci_get_drvdata(pci_dev);
	unsigned long flags;
	int err;

	DBG_SWITCH("bttv%d: resume\n", btv->c.nr);

	/* restore pci state */
	if (btv->state.disabled) {
		err=pci_enable_device(pci_dev);
		if (err) {
			DBG(KERN_WARNING "bttv%d: Can't enable device.\n",
								btv->c.nr);
			return err;
		}
		btv->state.disabled = 0;
	}
	err=pci_set_power_state(pci_dev, PCI_D0);
	if (err) {
		pci_disable_device(pci_dev);
		DBG(KERN_WARNING "bttv%d: Can't enable device.\n",
							btv->c.nr);
		btv->state.disabled = 1;
		return err;
	}

	pci_restore_state(pci_dev);

	/* restore bt878 state */
	bttv_reinit_tw680x(btv);
	gpio_inout(0xffffff, btv->state.gpio_enable);
	gpio_write(btv->state.gpio_data);

	/* restart dma */
	spin_lock_irqsave(&btv->s_lock,flags);
	btv->curr = btv->state.video;
	btv->cvbi = btv->state.vbi;

	// ST_0725
	// btv->loop_irq = btv->state.loop_irq;

	tw_buffer_activate_video(btv, &btv->curr);

	// ST_0622
	bttv_buffer_activate_vbi(btv, btv->cvbi);

	tw_DisableDMA( btv ) ;

	spin_unlock_irqrestore(&btv->s_lock,flags);
	
	return 0;
	
}


// ST_MODIFY_0922 for TW6801 support 
/*
static struct pci_device_id bttv_pci_tbl[] = {
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT848,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT849,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT878,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT879,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0,}
};
*/
static struct pci_device_id tw6801_pci_tbl[] = {

	{PCI_VENDOR_ID_TW6801, PCI_DEVICE_ID_TW6801,
	 PCI_ANY_ID,	PCI_ANY_ID, 	0, 0, 0 },
	{PCI_VENDOR_ID_TW6801, PCI_DEVICE_ID_TW6805,
	PCI_ANY_ID,	PCI_ANY_ID, 	0, 0, 0 },	 

	/* ST_0128, for tw6816 */
	{PCI_VENDOR_ID_TW6801, PCI_DEVICE_ID_TW6810,
	PCI_ANY_ID,	PCI_ANY_ID, 	0, 0, 0 },	 

	{PCI_VENDOR_ID_TW6801, PCI_DEVICE_ID_TW6811,
	PCI_ANY_ID,	PCI_ANY_ID, 	0, 0, 0 },	 
	
	{PCI_VENDOR_ID_TW6801, PCI_DEVICE_ID_TW6812,
	PCI_ANY_ID,	PCI_ANY_ID, 	0, 0, 0 },	 

	{PCI_VENDOR_ID_TW6801, PCI_DEVICE_ID_TW6813,
	PCI_ANY_ID,	PCI_ANY_ID, 	0, 0, 0 },	 

	/* ST_0128_DEL */
	/* 
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT848,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT849,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT878,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT879,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	 */
	 
	{0,}
};





// ST_MODIFY_0922 
// MODULE_DEVICE_TABLE(pci, bttv_pci_tbl);
MODULE_DEVICE_TABLE(pci, tw6801_pci_tbl);

// ST_MODIFY_0922 for TW6801 support 
/*
static struct pci_driver bttv_pci_driver = {
	.name     = "bttv",
	.id_table = bttv_pci_tbl,
	.probe    = bttv_probe,
	.remove   = __devexit_p(bttv_remove),
	.suspend  = bttv_suspend,
	.resume   = bttv_resume,
};
*/
static struct pci_driver TW6801_pci_driver = {
	.name	= 	"TW680x",
      	.id_table	= 	tw6801_pci_tbl,
      	.probe	= 	tw_probe,
      	.remove	=	__devexit_p(bttv_remove),
	.suspend =	bttv_suspend,
	.resume   =	bttv_resume,      	
};





static int tw_init_module(void)
{
	int ret;

	bttv_num = 0;

	if(switching) {
		printk( KERN_INFO "\n ========= TW68xx Linux Device Driver by ST, Version: 1.30.0128.1(Switching mode )======== \n" ) ;
	} else {
		printk( KERN_INFO "\n ========= TW68xx Linux Device Driver by ST, Version: 1.30.0128.1(Realtime mode ) ======== \n" ) ;
	}

	DBG(KERN_INFO "TW6801: driver version %d.%d.%d loaded\n",
	       (TW6801_VERSION_CODE >> 16) & 0xff,
	       (TW6801_VERSION_CODE >> 8) & 0xff,
	       TW6801_VERSION_CODE & 0xff);
	
#ifdef SNAPSHOT
	DBG(KERN_INFO "TW6801: snapshot date %04d-%02d-%02d\n",
	       SNAPSHOT/10000, (SNAPSHOT/100)%100, SNAPSHOT%100);
#endif

	// VIDEO_MAX_FRAME = 32

	if (gbuffers < 2 || gbuffers > VIDEO_MAX_FRAME)
		gbuffers = 2;

	if (gbufsize < 0 || gbufsize > TW6801_MAX_FBUF)
		gbufsize = TW6801_MAX_FBUF;
	

	gbufsize = (gbufsize + PAGE_SIZE - 1) & PAGE_MASK;

	if (bttv_verbose)
		DBG(KERN_INFO "TW6801: using %d buffers with %dk (%d pages) each for capture\n",
		       gbuffers, gbufsize >> 10, gbufsize >> PAGE_SHIFT);

	// ST_0413
	// tw_check_chipset();


// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	

	// ST_0129, Add for switching mode 
        ret = pci_register_driver(&TW6801_pci_driver);

	if (ret < 0) {
		DBG(KERN_WARNING "bttv: bus_register error: %d\n", ret);
        	bus_unregister(&TW6801_pci_driver);
	}

	return ret;

#else

	ret = bus_register(&bttv_sub_bus_type);
	
	if (ret < 0) {
		DBG(KERN_WARNING "bttv: bus_register error: %d\n", ret);
		return ret;
	}
	
	return pci_register_driver(&TW6801_pci_driver);

#endif


}

static void tw_cleanup_module(void)
{


	pci_unregister_driver(&TW6801_pci_driver);

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)	
	bus_unregister(&bttv_sub_bus_type);
#endif

	if(switching) {
		printk( KERN_INFO "\n ========= TW68xx Linux Device Driver clean up(Switching mode) ======== \n" ) ;
	} else {
		printk( KERN_INFO "\n ========= TW68xx Linux Device Driver clean up(Realtime mode) ======== \n" ) ;
	}
	return;
	
}

module_init( tw_init_module);
module_exit( tw_cleanup_module);

/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
